//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlglrc.cpp,v 1.4 2004/01/13 17:38:43 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

/* 

  @file
  @brief Implementation of GLRC
  
*/

#include "../base/gmlcommon.h"
#include <windows.h>
#include "gmlglrc.h"


using gml::GLRC;

//////////////////////////////////////////////////////////////////////////////
/// @param wnd window handle for which gl context should be created
/// 
GLRC::GLRC(HWND wnd)
{
  ASSERT(wnd);
  m_wnd = wnd;
  m_dc = ::GetDC(wnd);
  ASSERT(m_dc);
  m_created = false;
}

//////////////////////////////////////////////////////////////////////////////
///
GLRC::GLRC(HDC hdc)
{
  m_wnd = 0;
  m_dc = hdc;
  ASSERT(m_dc);
  m_created = false;
}

//////////////////////////////////////////////////////////////////////////////
/// @note Destroy() called inside destructor if was not called already
GLRC::~GLRC()
{
  if (m_created)
    Destroy();
}


//////////////////////////////////////////////////////////////////////////////
///
void GLRC::Destroy()
{
  wglDeleteContext(m_glrc);
  ::ReleaseDC(m_wnd, m_dc);
  m_created = false;
}

//////////////////////////////////////////////////////////////////////////////
///
bool GLRC::MakeCurrent()
{
  ASSERT(m_created);

  if (IsCurrent())
    return true;

  VERIFY(wglMakeCurrent(m_dc, m_glrc));

  return true;
}

//////////////////////////////////////////////////////////////////////////////
/// @note After creation, gl context is automatically become current
/// @return false in case of failure, true otherwise
bool GLRC::Create()
{
  ASSERT(!m_created);

  int nPixelFormat;

  DWORD flags;
  flags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;

  static PIXELFORMATDESCRIPTOR pfd =
  {
    sizeof(PIXELFORMATDESCRIPTOR),
    1,
    flags,
    PFD_TYPE_RGBA,
    24,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    32,
    0,
    0,
    PFD_MAIN_PLANE,
    0,
    0,
    0,
    0
  };

  pfd.cAlphaBits = 8;

  nPixelFormat = ChoosePixelFormat(m_dc, &pfd);

  VERIFY(SetPixelFormat(m_dc, nPixelFormat, &pfd));

  m_glrc = wglCreateContext(m_dc);

  return MakeCurrent();
}

//////////////////////////////////////////////////////////////////////////////
///
void GLRC::SwapBuffers()
{
  ASSERT(m_created);
  ::SwapBuffers(m_dc);
}

//////////////////////////////////////////////////////////////////////////////
///
bool GLRC::IsCurrent() const
{
  ASSERT(m_created);
  return ::wglGetCurrentContext() == m_glrc;
}

