//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlviewport.h,v 1.5 2004/01/13 17:38:43 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

/** @file gmlviewport.h
 *  @brief A class for 3d viewport
 * 
 */

#ifndef _GMLVIEWPORT_H_
#define _GMLVIEWPORT_H_


#include "gmlcamera.h"
#include "../math/gmlvector3.h"

namespace gml
{
  /** @addtogroup Utils3D
   * @{
   */

  /// Class representing 3d viewport (a window with accociated camera)
  class Viewport
  {
    public:
      Viewport() : m_x(0), m_y(0), m_xres(0), m_yres(0), m_camera(0)
      {
      }

      Viewport(int x, int y, int xres, int yres, Camera* camera = 0)
      {
        ASSERT(xres >= 0);
        ASSERT(yres >= 0);

        m_camera = camera;
        m_x = x;
        m_y = y;
        m_xres = xres;
        m_yres = yres;
      }

      void GetResolution(int& xres, int& yres) const
      {
        xres = m_xres; yres = m_yres;
      }
      void SetResolution(int xres, int yres)
      {
        m_xres = xres; m_yres = yres;
      }

      void SetOrigin(int x, int y)
      {
        m_x = x; m_y = y;
      }

      void GetOrigin(int& x, int& y) const
      {
        x = m_x; y = m_y;
      }


      const Matrix4x4d& GetTransform() const
      {
        ASSERT(m_camera);return m_camera->GetTransform();
      }

      void SetTransform(const Matrix4x4d& view_transform)
      {
        ASSERT(m_camera);m_camera->SetTransform(view_transform);
      }

      float GetViewAngle() const
      {
        ASSERT(m_camera); return m_camera->GetViewAngle();
      }

      void SetViewAngle(float view_angle)
      {
        ASSERT(m_camera); 
        ASSERT(view_angle >= 0); 
        ASSERT(view_angle <= 180);
        m_camera->SetViewAngle(view_angle);
      }

      const Camera* GetCamera() const
      {
        ASSERT(m_camera); return m_camera;
      }

      void SetCamera(Camera* cam)
      {
        m_camera = cam;
      }

      /// get view vector of camera (normalized)
      Vector3d GetViewVector() const
      {
        ASSERT(m_camera);
        return m_camera->GetViewVector();
      }

      /// get up vector of camera (normalized)
      Vector3d GetUpVector() const
      {
        ASSERT(m_camera);
        return m_camera->GetUpVector();
      }

      /// get up vector of camera (normalized)
      Vector3d GetRightVector() const
      {
        ASSERT(m_camera);
        return m_camera->GetRightVector();
      }

      /// get camera position
      Vector3d GetObserver() const
      {
        ASSERT(m_camera);
        return m_camera->GetObserver();
      }

      void SetObserver(const Vector3d& obs)
      {
        ASSERT(m_camera);
        m_camera->SetObserver(obs);
      }

      Matrix4x4d GetRotation() const
      {
        ASSERT(m_camera);
        return m_camera->GetRotation();
      }

      void SetRotation(const Matrix4x4d& rot)
      {
        ASSERT(m_camera);
        m_camera->SetRotation(rot);
      }

      void SetDistanceToTarget(float dist)
      {
        ASSERT(m_camera);
        m_camera->SetDistanceToTarget(dist);
      }

      float GetDistanceToTarget() const
      {
        ASSERT(m_camera);
        return m_camera->GetDistanceToTarget();
      }

    private:

      int m_x;
      int m_y;
      int m_xres;
      int m_yres;
      Camera* m_camera;
  };
  /** @} */
} // namespace gml

#endif