//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlframecounter.h,v 1.5 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

/** @file gmlframecounter.h
*  @brief Simple frame rate (FPS) counter
* 
*/
#ifndef _GMLFRAMECOUNTER_H
#define _GMLFRAMECOUNTER_H

namespace gml
{
  /** @addtogroup Utils3D
  * @{
  */

  /// Simple frame rate (FPS) counter
  class FrameCounter
  {
    public:
      /// constructor
      FrameCounter();
      /// destructor
      ~FrameCounter();

      /// Set counted update speed in milliseconds
      void SetUpdateSpeed(double ms)
      {
        ASSERT(ms > 0); m_update_speed = ms;
      }

      /// Return counter update speed
      double GetUpdateSpeed() const
      {
        return m_update_speed;
      }

      /// Set minimum number of frames needed to calculate FPS
      /**
       * FPS will be made and Updated() flag will be set only if
       * time since latest update > update speed and number of drawn frames
       * (calls to FrameFinished()) > minimum number of frames
       */
      void SetMinimumFrameCount(int count)
      {
        ASSERT(count > 0); m_min_frames = count;
      }

      /// Return counter update speed
      double GetMinimumFrameCount() const
      {
        return m_min_frames;
      }


      /// Signal that animation cycle is started
      /**
      * Probably should be called before first frame 
      */
      void Reset();

      /// Signal that current frame is to be drawn
      /**
      * Should be called before each scene display
      */
      void FrameStarted();

      /// Signal that current frame is finished
      /**
      * Should be called after each scene display
      */
      void FrameFinished();

      /// Tells whether FPS has been updated
      /**
       * Check this after call to FrameFinished. Flag cleared after next call to FrameFinished
       */
      bool Updated() const;

      /// Return the framecount (frames per second)
      double GetFrameCount() const;

      /// Return the mean frame time (msec)
      double GetMeanFrameTime() const;

    private:
      /// How many frames have been drawn since the last second
      int m_current_frames;
      /// How many frames have been drawn in the last second
      double m_frame_count;
      /// Last second's time count
      int m_last_second;
      /// Last update time count
      int m_last_update;
      /// update speed (ms)
      double m_update_speed;
      /// whether FPS has been updated
      bool m_updated;
      /// time for all frames within m_update_speed
      int m_total_frames_time;
      /// minimum number of frames to make FPS
      int m_min_frames;
      /// mean time of one frame
      double m_frame_time;
  };
  /** @} */
} // namespace gml

#endif
