//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlcamera.h,v 1.7 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

/** @file gmlcamera.h
 *  @brief A class for 3d virtual camera
 * 
 */

#ifndef _GMLCAMERA_H_
#define _GMLCAMERA_H_

#include "../math/gmlmatrix4.h"

namespace gml
{
  /** @addtogroup Utils3D
   * @{
   */

  /// Class representing 3d virtual camera
  class Camera
  {
    public:
      Camera()
      {
        m_view_transform.MakeIdentity();
        m_inv_view_transform.MakeIdentity();
        m_view_angle = 90;
        m_target_distance = 0;
      }
      /// set camera view angle (angle should be between 0 and 180 degrees)
      void SetViewAngle(float view_angle)
      {
        ASSERT(view_angle >= 0);
        ASSERT(view_angle <= 180);
        m_view_angle = view_angle;
      }
      /// return camera view angle
      float GetViewAngle() const
      {
        return m_view_angle;
      }

      /// set distance from camera to target
      void SetDistanceToTarget(float target_distance)
      {
        ASSERT(target_distance > 0);
        m_target_distance = target_distance;
      }

      /// return distance from camera to target
      float GetDistanceToTarget() const
      {
        return m_target_distance;
      }

      /// set view transformation
      const Matrix4x4d& GetTransform() const
      {
        return m_view_transform;
      }

      /// return view transformation
      void SetTransform(const Matrix4x4d& view_transform)
      {
        m_view_transform = view_transform;
        m_inv_view_transform = m_view_transform.Inverse();
      }

      /// get view vector of camera (normalized)
      Vector3d GetViewVector() const
      {
        Vector3d dir(0, 0, -1);
        m_inv_view_transform.MultMatrixDir(dir);
        return dir.Normalize();
      }

      /// get up vector of camera (normalized)
      Vector3d GetUpVector() const
      {
        Vector3d dir(0, 1, 0);
        m_inv_view_transform.MultMatrixDir(dir);
        return dir.Normalize();
      }

      /// get up vector of camera (normalized)
      Vector3d GetRightVector() const
      {
        Vector3d dir(1, 0, 0);
        m_inv_view_transform.MultMatrixDir(dir);
        return dir.Normalize();
      }

      // get camera position
      Vector3d GetObserver() const
      {
        return m_inv_view_transform.GetTranslation();
      }

      void SetObserver(const Vector3d& obs)
      {
        m_view_transform.SetTranslation(obs);
        m_inv_view_transform = m_view_transform.Inverse();
      }

      Matrix4x4d GetRotation() const
      {
        return m_view_transform.GetRotation();
      }

      void SetRotation(const Matrix4x4d& rot)
      {
        m_view_transform.SetRotation(rot);
        m_inv_view_transform = m_view_transform.Inverse();
      }

    protected:

      Matrix4x4d m_view_transform; ///< OGL ready viewing transform (global -> eye)
      Matrix4x4d m_inv_view_transform; ///< inverse view transform (eye -> global)
      float m_view_angle; ///< view angle (degrees)
      float m_target_distance; ///< distance to target
  };

  /** @} */
} // namespace gml

#endif
