//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlbrowser.h,v 1.7 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

/** @file gmlbrowser.h
 *  @brief A class for 3d virtual camera
 * 
 */

#ifndef _GMLBROWSER_H_
#define _GMLBROWSER_H_

#include "gmlviewport.h"
#include "../math/gmlvector2.h"
#include "../math/gmlbbox3.h"

namespace gml
{
  /** @addtogroup Utils3D
   * @{
   */

  class ModeBrowser
  {
    public:
      virtual ~ModeBrowser() = 0
      {
      };

      virtual void SetDomain(const BBox3f& scene_size,
                             bool update_position = true) = 0;
      virtual BBox3f GetDomain() const = 0;

      virtual void MouseButton(const Vector2i& pos, int button, bool pressed) = 0;
      virtual void MouseMotion(const Vector2i& pos, int button) = 0;
      virtual void MousePassiveMotion(const Vector2i& pos) = 0;
      virtual void MouseWheel(const Vector2i& pos, int delta) = 0;

      virtual bool KeyDown(int key, int options) = 0;
      virtual bool KeyUp(int key, int options) = 0;
  };

  class TrackballBrowser : public ModeBrowser
  {
    public:
      TrackballBrowser(Viewport& viewport);
      ~TrackballBrowser();

      void SetDomain(const BBox3f& scene_size, bool update_position = true);
      virtual BBox3f GetDomain() const
      {
        return m_scene_size;
      }

      virtual void MouseButton(const Vector2i& pos, int button, bool pressed);
      virtual void MouseMotion(const Vector2i& pos, int button);
      virtual void MousePassiveMotion(const Vector2i& pos);
      virtual void MouseWheel(const Vector2i& pos, int delta);

      virtual bool KeyDown(int key, int options)
      {
        return false;
      }
      virtual bool KeyUp(int key, int options)
      {
        return false;
      }

      // own interface
      void SetAspectMode(bool _aspect_mode)
      {
        aspect_mode = _aspect_mode;
      }

      bool AspectMode() const
      {
        return aspect_mode;
      }

      void SetDistance(int _dist)
      {
        dist = _dist;
      }
      int Distance() const
      {
        return dist;
      }

    private:
      void Reset();
      void SetTransform();
      void Point2Vector(int x, int y, int width, int height, Vector3d& v);

      void StartMotion(int x, int y);
      void StopMotion();
      void Motion(int x, int y);


    private:
      bool aspect_mode;
      int dist;

      Matrix4x4d m_transform;
      Vector3d m_axis;
      double m_angle;
      Viewport& m_viewport;
      int m_lasttime;
      Vector3d m_lastposition;

      BBox3f m_scene_size;
      Vector3d m_pivot_point;
      double m_distance;
  };

  class WalkBrowser : public ModeBrowser
  {
    public:
      WalkBrowser(Viewport& viewport);
      ~WalkBrowser();

      void SetDomain(const BBox3f& scene_size, bool update_position = true);
      virtual BBox3f GetDomain() const
      {
        return m_scene_size;
      }

      virtual void MouseButton(const Vector2i& pos, int button, bool pressed);
      virtual void MouseMotion(const Vector2i& pos, int button);
      virtual void MousePassiveMotion(const Vector2i& pos);
      virtual void MouseWheel(const Vector2i& pos, int delta);

      virtual bool KeyDown(int key, int options);
      virtual bool KeyUp(int key, int options);

    private:
      void UpdateTransform();
    private:

      Matrix4x4d m_transform;
      gml::Vector3d m_position;

      Viewport& m_viewport;
      BBox3f m_scene_size;
      float m_step_size;
      int m_x_old, m_y_old;
      int m_x_press, m_y_press;
  };

  /// 3D navigation helper class
  class Browser
  {
    public:
      enum NAV_MODE { NAV_TRACKBALL, NAV_WALK };

      Browser(Viewport& viewport);

      void SetNavigationMode(NAV_MODE nav_mode)
      {
        m_nav_mode = nav_mode;
        switch (m_nav_mode)
        {
          case NAV_TRACKBALL:
            m_cur_browser = &m_trackball_browser;
            break;
          case NAV_WALK:
            m_cur_browser = &m_walkbrowser;
            break;
        };
      }
      ModeBrowser* GetModeBrowser()
      {
        return m_cur_browser;
      }

      void SetDomain(const BBox3f& scene_size, bool update_position = true)
      {
        m_cur_browser->SetDomain(scene_size, update_position);
      }

      BBox3f GetDomain() const
      {
        return m_cur_browser->GetDomain();
      }

      void MouseButton(const Vector2i& pos, int button, bool pressed)
      {
        m_cur_browser->MouseButton(pos, button, pressed);
      }

      void MouseMotion(const Vector2i& pos, int button)
      {
        m_cur_browser->MouseMotion(pos, button);
      }

      void MousePassiveMotion(const Vector2i& pos)
      {
        m_cur_browser->MousePassiveMotion(pos);
      }

      void MouseWheel(const Vector2i& pos, int delta)
      {
        m_cur_browser->MouseWheel(pos, delta);
      }

      bool KeyDown(int key, int options)
      {
        return m_cur_browser->KeyDown(key, options);
      }
      bool KeyUp(int key, int options)
      {
        return m_cur_browser->KeyUp(key, options);
      }
    private:
      ModeBrowser* m_cur_browser;
      TrackballBrowser m_trackball_browser;
      WalkBrowser m_walkbrowser;

      NAV_MODE m_nav_mode;
      Viewport& m_viewport;
  };



  /** @} */
} // namespace gml

#endif
