//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlbrowser.cpp,v 1.8 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.


#include "../base/gmlcommon.h"
#include "../math/gmlquat.h"
#include "gmlbrowser.h"
#include <math.h>

using namespace gml;

Browser::Browser(Viewport& viewport) : m_viewport(viewport),
                                       m_trackball_browser(viewport),
                                       m_walkbrowser(viewport)
{
  m_nav_mode = NAV_TRACKBALL;
  m_cur_browser = &m_trackball_browser;
}


TrackballBrowser::TrackballBrowser(Viewport& viewport) : m_viewport(viewport),
                                                         dist(0),
                                                         aspect_mode(false)
{
  Reset();
}

TrackballBrowser::~TrackballBrowser()
{
}

void TrackballBrowser::Reset()
{
  /* put the identity in the trackball transform */
  m_transform.MakeIdentity();
  m_angle = 0;
  m_axis.SetValue(0, 1, 0);
  m_pivot_point.SetValue(0);
  m_distance = 0;
  m_scene_size.vmax = m_scene_size.vmin = Vector3f(0, 0, 0);
}

void TrackballBrowser::SetDomain(const BBox3f& scene_size,
                                 bool update_position /*= true*/)
{
  m_scene_size = scene_size;

  if (!update_position)
    return;

  m_pivot_point = ConvD(m_scene_size.Center());
  m_distance = m_scene_size.Diag().Length() * 1.2;
  SetTransform();
}
void TrackballBrowser::MouseButton(const Vector2i& pos,
                                   int button,
                                   bool pressed)
{
  if (button == 0)
    if (pressed)
    {
      StartMotion(pos.x, pos.y);
    }
    else
      StopMotion();
}
void TrackballBrowser::MouseMotion(const Vector2i& pos, int button)
{
  if (button == 0)
    Motion(pos.x, pos.y);
}

void TrackballBrowser::MousePassiveMotion(const Vector2i& pos)
{
  /* nothing */
}

void TrackballBrowser::MouseWheel(const Vector2i& pos, int delta)
{
  m_distance -= delta * m_scene_size.Diag().Length() / 1000.0;
  if (m_distance < 0)
    m_distance = 0;
  SetTransform();
}

void TrackballBrowser::SetTransform()
{
  Quaterniond rot(m_axis, ToRadians(m_angle));
  Matrix4x4d res;
  rot.GetValue(res);

  m_transform.MultLeft(res); // finished with rotation  transform

  Matrix4x4d final_transform(1);
  final_transform.SetTranslation(Vector3d(0, 0, -m_distance));

  final_transform.MultRight(m_transform);


  Matrix4x4d pivot_matrix(1);
  pivot_matrix.SetTranslation(-m_pivot_point);

  final_transform.MultRight(pivot_matrix);

  /*
    glTranslatef(0,0,-m_distance); // distance 
  glMultMatrixf(m_trackball.Transform().get_value());
  glTranslatef(-m_pivot[0],-m_pivot[1], -m_pivot[2]); // pivot point*/

  m_viewport.SetTransform(final_transform);
}


void TrackballBrowser::Point2Vector(int x,
                                    int y,
                                    int width,
                                    int height,
                                    Vector3d& v)
{
  double d, a;

  if (!aspect_mode)
  {
    /* project x, y onto a hemi-sphere centered within width, height. */
    v[0] = (2.0 * x - width) / width;
    v[1] = (height - 2.0 * y) / height;
  }
  else
  {
    int th = dist;
    int r = gml::Min(width, height) - th * 2;
    v[0] = (2.0 * x - width) / r;
    v[1] = (height - y * 2.0) / r;
  }
  d = sqrt(v[0] * v[0] + v[1] * v[1]);
  v[2] = cos((3.14159265 / 2.0) * ((d < 1.0) ? d : 1.0));
  a = 1.0 / sqrt(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);
  v[0] *= a;
  v[1] *= a;
  v[2] *= a;
}


void TrackballBrowser::StartMotion(int x, int y)
{
  int width, height;
  m_viewport.GetResolution(width, height);
  //m_lasttime = GetTickCount();
  Point2Vector(x, y, width, height, m_lastposition);
}

void TrackballBrowser::StopMotion()
{
  m_angle = 0.0;
}

void TrackballBrowser::Motion(int x, int y)
{
  int width, height;
  m_viewport.GetResolution(width, height);

  Vector3d current_position;
  Vector3d d;

  Point2Vector(x, y, width, height, current_position);

  /* calculate the angle to rotate by (directly proportional to the
  length of the mouse movement */
  d = current_position - m_lastposition;
  m_angle = 90.0 * sqrt(d.SqrLength());
  m_axis = CrossProduct(m_lastposition, current_position);
#if 0
  /* calculate the axis of rotation (cross product) */
  tb_axis[0] = tb_lastposition[1] * current_position[2] - 
    tb_lastposition[2] * current_position[1];
  tb_axis[1] = tb_lastposition[2] * current_position[0] - 
    tb_lastposition[0] * current_position[2];
  tb_axis[2] = tb_lastposition[0] * current_position[1] - 
    tb_lastposition[1] * current_position[0];
#endif

  /* reset for next time */
  //m_lasttime = GetTickCount();

  m_lastposition = current_position;


  SetTransform();
}

// =========================== Walk browser ============================

WalkBrowser::WalkBrowser(Viewport& viewport) : m_viewport(viewport),
                                               m_transform(1),
                                               m_position(0,
                                                          0,
                                                          0),
                                               m_step_size(10)
{
}
WalkBrowser::~WalkBrowser()
{
}

void WalkBrowser::SetDomain(const BBox3f& scene_size,
                            bool update_position /*= true*/)
{
  m_scene_size = scene_size;

  if (!update_position)
    return;

  double diag_length = m_scene_size.Diag().Length();
  m_step_size = (float) diag_length * 2;

  m_position.SetValue(0, 0, diag_length);
  m_transform.Identity();
  //m_transform.SetTranslation(m_position);
  UpdateTransform();
  //m_viewport.SetTransform(m_transform);
}

void WalkBrowser::MouseButton(const Vector2i& pos, int button, bool pressed)
{
  if (pressed)
  {
    m_x_press = pos.x;
    m_y_press = pos.y;
  }
}
void WalkBrowser::MouseMotion(const Vector2i& pos, int button)
{
  int view_width = -1, view_height = -1;
  m_viewport.GetResolution(view_width, view_height);
  int vp_diag = (int) sqrt((double) view_width* view_width +
                           ((double) view_height* view_height));

  // absolute values of rotation
  double val_up = (double) (m_x_old - pos.x) / vp_diag;
  double val_right = (double) (m_y_old - pos.y) / vp_diag;

  //double val_right_motion = (double)(m_y_press - pos.y) / vp_diag;

  gml::Vector3d viewpos = m_viewport.GetObserver();

  gml::Quaterniond rot(gml::Vector3d(0, 1, 0), -val_up * 4);
  Matrix4x4d res;
  rot.GetValue(res);

  m_transform.MultLeft(res); // finished with rotation  transform


  Vector3d dir(0, 0, -1);
  m_transform.MultDirMatrix(dir); // get current viewdirection

  m_position += dir * (val_right) * m_step_size/** CAMERA_MOTION_SPEED*/;

  UpdateTransform();

  m_x_old = pos.x;
  m_y_old = pos.y;
}

void WalkBrowser::UpdateTransform()
{
  Matrix4x4d translate(1);
  translate.SetTranslation(m_position);

  Matrix4x4d translate1(1);
  translate.SetTranslation(-m_position);

  Matrix4x4d final_transform = translate1* m_transform* translate;
  m_viewport.SetTransform(final_transform);
}

void WalkBrowser::MousePassiveMotion(const Vector2i& pos)
{
  m_x_old = pos.x;
  m_y_old = pos.y;
}
void WalkBrowser::MouseWheel(const Vector2i& pos, int delta)
{
}

bool WalkBrowser::KeyDown(int key, int options)
{
  switch (key)
  {
    case 'a':
    case 'A':
      m_position -= m_viewport.GetRightVector() * m_step_size / 20;
      UpdateTransform();
      return true;

    case 'd':
    case 'D':
      m_position += m_viewport.GetRightVector() * m_step_size / 20;
      UpdateTransform();
      return true;

    case 'w':
    case 'W':
      m_position += m_viewport.GetUpVector() * m_step_size / 20;
      UpdateTransform();
      return true;

    case 's':
    case 'S':
      m_position -= m_viewport.GetUpVector() * m_step_size / 20;
      UpdateTransform();
      return true;

    default:
      break;
  };
  return false;
}
bool WalkBrowser::KeyUp(int key, int options)
{
  return false;
}
