//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlsimpleimageloader.cpp,v 1.26 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#include "../../base/gmlcommon.h"
#include "../bitmap/gmlimage.h"
#include "gmlsimpleimageloader.h"
#include "../../files/gmlpathstring.h"

#include <png.h>

extern "C"
{
#include <jpeglib.h>
}

bool gml::SimpleImageLoader::EnumerateLoadableFormats(void (*f)
                                                      (IMAGE_FORMAT format))
{
  f(FORMAT_BMP);
  f(FORMAT_JPEG); 
  f(FORMAT_PNG);
  return true;
}

bool gml::SimpleImageLoader::EnumerateSaveableFormats(void (*f)
                                                      (IMAGE_FORMAT format))
{
  return true;
}

gml::SimpleImageLoader::IMAGE_FORMAT gml::SimpleImageLoader::CheckBitMapFile(const std::string& path)
{
  /*
  std::locale a;
  std::string s = a.name();
  const std::collate<char>& fac = std::use_facet<std::collate<char> >(a, 0, true);
  const std::ctype<char>& type = std::use_facet<std::ctype<char> >(a, 0, true);
  std::string test1("this is a test");
  type.toupper(test1.begin(), test1.end());*/
  gml::PathString path1(path);
  if (stricmp(path1.Extension().c_str(), "jpg") == 0 ||
      stricmp(path1.Extension().c_str(),
                                                                "jpeg") == 0)
    return FORMAT_JPEG;

  if (stricmp(path1.Extension().c_str(), "png") == 0)
    return FORMAT_PNG;

  if (stricmp(path1.Extension().c_str(), "bmp") == 0)
    return FORMAT_BMP;

  return FORMAT_UNKNOWN;
}

bool gml::SimpleImageLoader::LoadBitMap(const std::string& path,
                                        Image& out_bitmap,
                                        IMAGE_FORMAT format /* = FORMAT_UNKNOWN */,
                                        int flags /* = 0 */)
{
  IMAGE_FORMAT f = format;
  if (f == FORMAT_UNKNOWN)
    f = CheckBitMapFile(path);

  //   EXIF
  m_pcJpegAPP1.clear();

  switch (f)
  {
    case FORMAT_JPEG:
      return LoadJPG(path, out_bitmap, flags & PRESERVE_JPEG_EXIF);
    case FORMAT_PNG:
      return LoadPNG(path, out_bitmap);
    case FORMAT_BMP:
      return LoadBMP(path, out_bitmap);
    default:
      return false;
  }
  return false;
}

bool gml::SimpleImageLoader::SaveBitMap(const std::string& path,
                                        const Image& bitmap,
                                        IMAGE_FORMAT format,
                                        int in_iQuality,
                                        int flags /*= 0*/)
{
  if (format == FORMAT_UNKNOWN)
    format = CheckBitMapFile(path);

  switch (format)
  {
    case FORMAT_JPEG:
      return SaveJPG(path, bitmap, in_iQuality, flags & PRESERVE_JPEG_EXIF);

    case FORMAT_PNG:
      return SavePNG(path, bitmap);

    case FORMAT_BMP:
      return SaveBMP(path, bitmap);

    default:
      return false;
  }

  return false;
}

bool gml::SimpleImageLoader::LoadPNG(const std::string& path,
                                     Image& out_bitmap)
{
  png_structp png_ptr;
  png_infop info_ptr;
  unsigned int sig_read = 0;
  FILE* fp;

  if ((fp = fopen(path.c_str(), "rb")) == NULL)
    return false;

  png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, 0, 0, 0);

  if (png_ptr == NULL)
  {
    fclose(fp);
    return false;
  }

  /* Allocate/initialize the memory for image information.  REQUIRED. */
  info_ptr = png_create_info_struct(png_ptr);
  if (info_ptr == NULL)
  {
    fclose(fp);
    png_destroy_read_struct(&png_ptr, png_infopp_NULL, png_infopp_NULL);
    return false;
  }

  if (setjmp(png_jmpbuf(png_ptr)))
  {
    /* Free all of the memory associated with the png_ptr and info_ptr */
    png_destroy_read_struct(&png_ptr, &info_ptr, png_infopp_NULL);
    fclose(fp);
    /* If we get here, we had a problem reading the file */
    return false;
  }

  /* Set up the input control if you are using standard C streams */
  png_init_io(png_ptr, fp);

  png_read_png(png_ptr, info_ptr, PNG_TRANSFORM_SWAP_ENDIAN, png_voidp_NULL);


  // now create gml image
  gml::Image::REPRES r;
  gml::Image::FORMAT f;

  switch (info_ptr->color_type)
  {
    case PNG_COLOR_TYPE_GRAY:
      f = gml::Image::F_L;
      break;
    case PNG_COLOR_TYPE_RGB:
      f = gml::Image::F_RGB;
      break;
    case PNG_COLOR_TYPE_RGBA:
      f = gml::Image::F_RGBA;
      break;

    default:
      ASSERT(false);
  }

  switch (info_ptr->bit_depth)
  {
    case 8:
      r = gml::Image::R_BYTE;
      break;
    case 16:
      r = gml::Image::R_WORD;
      break;
    default:
      ASSERT(false);
  }

  out_bitmap.Create(info_ptr->width, info_ptr->height, f, r);
  if (!out_bitmap.Created())
  {
    /* close the file */
    fclose(fp);
    return false;
  }

  BYTE** dst_lines = out_bitmap.GetLineArray();
  BYTE** src_lines = info_ptr->row_pointers;

  //out_bitmap.ChangeOrientation(gml::Image::O_TOPLEFT);

  for (unsigned int y = 0; y < info_ptr->height; ++y)
  {
    BYTE* dst_line = dst_lines[y];
    BYTE* src_line = src_lines[info_ptr->height - y - 1];
    memcpy(dst_line,
           src_line,
           out_bitmap.GetWidth() * gml::Image::Format2Channels(f));//3);
  }

  /* clean up after the read, and free any memory allocated - REQUIRED */
  png_destroy_read_struct(&png_ptr, &info_ptr, png_infopp_NULL);

  /* close the file */
  fclose(fp);

  return true;
}

static jmp_buf g_ErrorRestart;              

static void restart_error_exit(j_common_ptr pJInfo)
{
  longjmp(g_ErrorRestart, -2);
}


bool gml::SimpleImageLoader::LoadJPG(const std::string& path,
                                     Image& out_bitmap,
                                     bool in_bReadEXIF)
{
  /* This struct contains the JPEG decompression parameters and pointers to
   * working space (which is allocated as needed by the JPEG library).
   */
  struct jpeg_decompress_struct cinfo;
  /* We use our private extension JPEG error handler.
   * Note that this struct must live as long as the main JPEG parameter
   * struct, to avoid dangling-pointer problems.
   */
  //struct my_error_mgr jerr;

  /* More stuff */
  FILE* infile;   /* source file */
  JSAMPARRAY buffer;    /* Output row buffer */
  int row_stride;   /* physical row width in output buffer */

  /* In this example we want to open the input file before doing anything else,
   * so that the setjmp() error recovery below can assume the file is open.
   * VERY IMPORTANT: use "b" option to fopen() if you are on a machine that
   * requires it in order to read binary files.
   */

  if ((infile = fopen(path.c_str(), "rb")) == NULL)
  {
    return false;
  }

  /* Step 1: allocate and initialize JPEG decompression object */


  /* We set up the normal JPEG error routines, then override error_exit. */

  struct jpeg_error_mgr jerr;
  cinfo.err = jpeg_std_error(&jerr);
  cinfo.err->error_exit = &restart_error_exit;            

  if (setjmp(g_ErrorRestart))
  {
    /* If we get here, the JPEG code has signaled an error.
     * We need to clean up the JPEG object, close the input file, and return.
     */
    jpeg_destroy_decompress(&cinfo);
    fclose(infile);
    return false;
  }

  /* Now we can initialize the JPEG decompression object. */
  jpeg_create_decompress(&cinfo);

  /* Step 2: specify data source (eg, a file) */

  jpeg_stdio_src(&cinfo, infile);

  if (in_bReadEXIF)
    //    EXIF
    jpeg_save_markers(&cinfo, JPEG_APP0 + 1, 0xFFFF);

  /* Step 3: read file parameters with jpeg_read_header() */

  (void) jpeg_read_header(&cinfo, TRUE);
  /* We can ignore the return value from jpeg_read_header since
   *   (a) suspension is not possible with the stdio data source, and
   *   (b) we passed TRUE to reject a tables-only JPEG file as an error.
   * See libjpeg.doc for more info.
   */


  /* Step 4: set parameters for decompression */

  /* In this example, we don't need to change any of the defaults set by
   * jpeg_read_header(), so we do nothing here.
   */

  cinfo.out_color_space = JCS_RGB;

  /* Step 5: Start decompressor */

  (void) jpeg_start_decompress(&cinfo);
  /* We can ignore the return value since suspension is not possible
   * with the stdio data source.
   */

  // now create gml image
  gml::Image::REPRES r = gml::Image::R_BYTE;
  gml::Image::FORMAT f = gml::Image::F_RGB;
  ASSERT(cinfo.output_components == 3);

  out_bitmap.Create(cinfo.output_width, cinfo.output_height, f, r);
  if (!out_bitmap.Created())
  {
    /* close the file */
    fclose(infile);
    return false;
  }

  BYTE** dst_lines = out_bitmap.GetLineArray();

  /* We may need to do some setup of our own at this point before reading
   * the data.  After jpeg_start_decompress() we have the correct scaled
   * output image dimensions available, as well as the output colormap
   * if we asked for color quantization.
   * In this example, we need to make an output work buffer of the right size.
   */ 
  /* JSAMPLEs per row in output buffer */
  row_stride = cinfo.output_width * cinfo.output_components;
  /* Make a one-row-high sample array that will go away when done with image */
  buffer = (*cinfo.mem->alloc_sarray)
           ((j_common_ptr) & cinfo, JPOOL_IMAGE, row_stride, 1);

  /* Step 6: while (scan lines remain to be read) */
  /*           jpeg_read_scanlines(...); */

  /* Here we use the library's state variable cinfo.output_scanline as the
   * loop counter, so that we don't have to keep track ourselves.
   */
  while (cinfo.output_scanline < cinfo.output_height)
  {
    /* jpeg_read_scanlines expects an array of pointers to scanlines.
     * Here the array is only one element long, but you could ask for
     * more than one scanline at a time if that's more convenient.
     */
    (void) jpeg_read_scanlines(&cinfo, buffer, 1);
    /* Assume put_scanline_someplace wants a pointer and sample count. */
    BYTE* dst_line = dst_lines[cinfo.output_height - cinfo.output_scanline];
    BYTE* src_line = buffer[0];
    memcpy(dst_line, src_line, row_stride);
  }

  ///   EXIF
  if (in_bReadEXIF)
  {
    jpeg_saved_marker_ptr marker;

    for (marker = cinfo.marker_list; marker != NULL; marker = marker->next)
    {
      if (marker->marker != (JPEG_APP0 + 1))
        continue;     //    APP1

      m_pcJpegAPP1.resize(marker->data_length);
      memcpy(&m_pcJpegAPP1[0], marker->data, marker->data_length);
      break;
    }


    /* Step 7: Finish decompression */

    (void) jpeg_finish_decompress(&cinfo);
    /* We can ignore the return value since suspension is not possible
     * with the stdio data source.
     */
  }
  /* Step 8: Release JPEG decompression object */

  /* This is an important step since it will release a good deal of memory. */
  jpeg_destroy_decompress(&cinfo);

  /* After finish_decompress, we can close the input file.
   * Here we postpone it until after no more JPEG errors are possible,
   * so as to simplify the setjmp error logic above.  (Actually, I don't
   * think that jpeg_destroy can do an error exit, but why assume anything...)
   */
  fclose(infile);

  /* At this point you may want to check to see whether any corrupt-data
   * warnings occurred (test whether jerr.pub.num_warnings is nonzero).
   */

  /* And we're done! */
  return 1;
}

//////////////////////////////////////////////////////////////////////////////////
//
bool gml::SimpleImageLoader::SavePNG(const std::string& path,
                                     const Image& out_bitmap)
{
  png_structp png_ptr;
  png_infop info_ptr;
  unsigned int sig_read = 0;
  unsigned int iYStart, iYEnd, iYStep;

  FILE* fp;

  if ((fp = fopen(path.c_str(), "wb")) == NULL)
    return false;

  png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, 0, 0, 0);

  if (png_ptr == NULL)
  {
    fclose(fp);
    return false;
  }

  /* Allocate/initialize the memory for image information.  REQUIRED. */
  info_ptr = png_create_info_struct(png_ptr);
  if (info_ptr == NULL)
  {
    fclose(fp);
    png_destroy_write_struct(&png_ptr, png_infopp_NULL);
    return false;
  }

  if (setjmp(png_jmpbuf(png_ptr)))
  {
    /* Free all of the memory associated with the png_ptr and info_ptr */
    png_destroy_write_struct(&png_ptr, &info_ptr);
    fclose(fp);
    /* If we get here, we had a problem reading the file */
    return false;
  }

  /* Set up the input control if you are using standard C streams */
  png_init_io(png_ptr, fp);


  // Set compression level for png file
  png_set_compression_level(png_ptr, Z_BEST_COMPRESSION);

  png_set_compression_mem_level(png_ptr, 8);
  png_set_compression_strategy(png_ptr, Z_DEFAULT_STRATEGY);
  png_set_compression_window_bits(png_ptr, 15);
  png_set_compression_method(png_ptr, 8);
  png_set_compression_buffer_size(png_ptr, 8192);

  int color_type, bit_depth;

  switch (out_bitmap.GetFormat())
  {
    case gml::Image::F_L:
      color_type = PNG_COLOR_TYPE_GRAY;
      break;
    case gml::Image::F_RGB:
      color_type = PNG_COLOR_TYPE_RGB;
      break;
    case gml::Image::F_RGBA:
      color_type = PNG_COLOR_TYPE_RGB_ALPHA;
      break;
    default:
      ASSERT(false);
      return false;
  }

  switch (out_bitmap.GetRepres())
  {
    case gml::Image::R_BYTE:
      bit_depth = 8;
      break;
    case gml::Image::R_WORD:
      bit_depth = 16;
      break;
    default:
      ASSERT(false);
  }

  png_set_IHDR(png_ptr,
               info_ptr,
               out_bitmap.GetWidth(),
               out_bitmap.GetHeight(),
               bit_depth,
               color_type,
               PNG_INTERLACE_NONE,
               PNG_COMPRESSION_TYPE_DEFAULT,
               PNG_FILTER_TYPE_DEFAULT);

  //out_bitmap.Lock(true, false);

  BYTE** src_lines = new BYTE* [out_bitmap.GetHeight()];
  BYTE** img_lines = out_bitmap.GetLineArray();

  if (out_bitmap.GetOrient() == gml::Image::O_BOTTOMLEFT)
  {
    iYStart = out_bitmap.GetHeight() - 1;
    iYEnd = -1;
    iYStep = -1;
  }
  else
  {
    iYStart = 0;
    iYEnd = out_bitmap.GetHeight();
    iYStep = 1;
  }

  for (int y = iYStart, iY = 0; y != iYEnd; y += iYStep, iY++)
  {
    src_lines[iY] = img_lines[y];
  }

  png_set_rows(png_ptr, info_ptr, src_lines);

  png_write_png(png_ptr, info_ptr, PNG_TRANSFORM_SWAP_ENDIAN, png_voidp_NULL);

  //out_bitmap.Unlock();

  /* clean up after the read, and free any memory allocated - REQUIRED */
  png_destroy_write_struct(&png_ptr, &info_ptr);

  /* close the file */
  fclose(fp);

  return true;
}

//////////////////////////////////////////////////////////////////////////////////
//
bool gml::SimpleImageLoader::SaveJPG(const std::string& path,
                                     const Image& out_bitmap,
                                     int in_iQuality,
                                     bool in_bSaveEXIF)
{
  if (in_iQuality < 0)
    in_iQuality = 0;

  if (in_iQuality > 100)
    in_iQuality = 100;

  // Check format
  if (out_bitmap.GetFormat() != gml::Image::F_RGB ||
      out_bitmap.GetRepres() != gml::Image::R_BYTE)
    return false; //not supported

  /* This struct contains the JPEG compression parameters and pointers to
   * working space (which is allocated as needed by the JPEG library).
   * It is possible to have several such structures, representing multiple
   * compression/decompression processes, in existence at once.  We refer
   * to any one struct (and its associated working data) as a "JPEG object".
   */
  struct jpeg_compress_struct cinfo;
  /* This struct represents a JPEG error handler.  It is declared separately
   * because applications often want to supply a specialized error handler
   * (see the second half of this file for an example).  But here we just
   * take the easy way out and use the standard error handler, which will
   * print a message on stderr and call exit() if compression fails.
   * Note that this struct must live as long as the main JPEG parameter
   * struct, to avoid dangling-pointer problems.
   */
  struct jpeg_error_mgr jerr;
  /* More stuff */
  FILE* outfile;    /* target file */
  JSAMPROW row_pointer[1];  /* pointer to JSAMPLE row[s] */
  int row_stride;   /* physical row width in image buffer */

  /* Step 1: allocate and initialize JPEG compression object */

  /* We have to set up the error handler first, in case the initialization
   * step fails.  (Unlikely, but it could happen if you are out of memory.)
   * This routine fills in the contents of struct jerr, and returns jerr's
   * address which we place into the link field in cinfo.
   */
  cinfo.err = jpeg_std_error(&jerr);
  /* Now we can initialize the JPEG compression object. */
  jpeg_create_compress(&cinfo);

  /* Step 2: specify data destination (eg, a file) */
  /* Note: steps 2 and 3 can be done in either order. */

  /* Here we use the library-supplied code to send compressed data to a
   * stdio stream.  You can also write your own code to do something else.
   * VERY IMPORTANT: use "b" option to fopen() if you are on a machine that
   * requires it in order to write binary files.
   */
  if ((outfile = fopen(path.c_str(), "wb")) == NULL)
  {
    return false;
  }

  jpeg_stdio_dest(&cinfo, outfile);

  /* Step 3: set parameters for compression */

  /* First we supply a description of the input image.
   * Four fields of the cinfo struct must be filled in:
   */
  cinfo.image_width = out_bitmap.GetWidth();  /* image width and height, in pixels */
  cinfo.image_height = out_bitmap.GetHeight();
  cinfo.input_components = 3;   /* # of color components per pixel */
  cinfo.in_color_space = JCS_RGB;   /* colorspace of input image */
  /* Now use the library's routine to set default compression parameters.
   * (You must set at least cinfo.in_color_space before calling this,
   * since the defaults depend on the source color space.)
   */
  jpeg_set_defaults(&cinfo);
  /* Now you can set any non-default parameters you wish to.
   * Here we just illustrate the use of quality (quantization table) scaling:
   */
  jpeg_set_quality(&cinfo,
                   in_iQuality,
                   TRUE /* limit to baseline-JPEG values */);

  /* Step 4: Start compressor */

  /* TRUE ensures that we will write a complete interchange-JPEG file.
   * Pass TRUE unless you are very sure of what you're doing.
   */
  jpeg_start_compress(&cinfo, TRUE);


  if (in_bSaveEXIF && m_pcJpegAPP1.size() > 0)
  {
    //  APPn    JPEG' (  ) TODO
    jpeg_write_marker(&cinfo,
                      JPEG_APP0 + 1,
                      &m_pcJpegAPP1[0],
                      m_pcJpegAPP1.size());
  }

  /* Step 5: while (scan lines remain to be written) */
  /*           jpeg_write_scanlines(...); */

  /* Here we use the library's state variable cinfo.next_scanline as the
   * loop counter, so that we don't have to keep track ourselves.
   * To keep things simple, we pass one scanline per call; you can pass
   * more if you wish, though.
   */
  row_stride = out_bitmap.GetWidth() * 3; /* JSAMPLEs per row in image_buffer */

  if (out_bitmap.GetOrient() == gml::Image::O_TOPLEFT)
  {
    while (cinfo.next_scanline < cinfo.image_height)
    {
      /* jpeg_write_scanlines expects an array of pointers to scanlines.
       * Here the array is only one element long, but you could pass
       * more than one scanline at a time if that's more convenient.
       */
      row_pointer[0] = out_bitmap.GetLineArray()[cinfo.next_scanline];
      (void) jpeg_write_scanlines(&cinfo, row_pointer, 1);
    }
  }
  else
  {
    while (cinfo.next_scanline < cinfo.image_height)
    {
      /* jpeg_write_scanlines expects an array of pointers to scanlines.
       * Here the array is only one element long, but you could pass
       * more than one scanline at a time if that's more convenient.
       */
      row_pointer[0] = out_bitmap.GetLineArray()[out_bitmap.GetHeight() - 1 - cinfo.next_scanline];
      (void) jpeg_write_scanlines(&cinfo, row_pointer, 1);
    }
  }


  /* Step 6: Finish compression */

  jpeg_finish_compress(&cinfo);
  /* After finish_compress, we can close the output file. */
  fclose(outfile);

  /* Step 7: release JPEG compression object */

  /* This is an important step since it will release a good deal of memory. */
  jpeg_destroy_compress(&cinfo);

  /* And we're done! */

  return true;
}



//===================================================================
//= Function name : gml::SimpleImageLoader::GetExifMarker
//= Description   : 
//= Return type   : void 
//===================================================================

void gml::SimpleImageLoader::GetExifMarker(std::vector <unsigned char>& out_pcData)
{
  out_pcData = m_pcJpegAPP1;
}


//===================================================================
//= Function name : gml::SimpleImageLoader::SetExifMarker
//= Description   : 
//= Return type   : void 
//===================================================================

void gml::SimpleImageLoader::SetExifMarker(std::vector <unsigned char>& in_pcData)
{
  m_pcJpegAPP1 = in_pcData;
}
