//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlbmp.cpp,v 1.26 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#include "../../base/gmlcommon.h"
#include "gmlsimpleimageloader.h"
#include "../../files/gmlfile.h"
#include "../bitmap/gmlimage.h"


#define BMP_SIZE_FILEHEADER 14  // size of the BMP file header
#define BMP_SIZE_INFOHEADER 40  // size of the BMP file header of information

#define BMP_COLOR_BITS_01  1 
#define BMP_COLOR_BITS_04  4 
#define BMP_COLOR_BITS_08  8 
#define BMP_COLOR_BITS_24 24 

namespace gml
{
  struct RGBQUAD
  {
      BYTE blue;        // relative intensity of blu color
      BYTE green;       // relative intensity of green color
      BYTE red;         // relative intensity of red color
      BYTE reserved;    // not used

      bool operator ==(const RGBQUAD& src) const;
  };

  bool RGBQUAD::operator ==(const RGBQUAD& src) const
  {
    return ((blue == src.blue) &&
            (green == src.green) &&
            (red == src.red) &&
            (reserved == src.reserved));
  }
};
static unsigned int uInt16Number(BYTE buf[2]);
static unsigned int uInt32Number(BYTE buf[4]);
static void IntTo2Bytes(int val, BYTE buf[2]);
static void IntTo4Bytes(int val, BYTE buf[4]);
static int rgb_buffer_size(int xres, int type_rgb);
static int HalfOfByte(BYTE val, int first_last);
static void BitsOfByte(BYTE val, int* bits); 

bool ReadFileHeader(gml::File& file, int& bitmap_pos)
{
  // Set pointer to the begin of file
  if (!file.Seek(0))
    return false;

  // Read header of the file (BITMAPHEADER)
  BYTE header[BMP_SIZE_FILEHEADER];
  int numb = 0;  
  file.Read(header, BMP_SIZE_FILEHEADER, numb);
  if (numb != BMP_SIZE_FILEHEADER)
    return false;

  // First two bytes must be 'B' and 'M'
  if (header[0] != 'B' || header[1] != 'M')
    return false;

  //              b[2]-b[5] size of the file
  // Really it is not size of the file, for example
  // for the resolution 225x329 and for the 8-bits (256 colors) 
  // per pixel the size calculated by the formula:
  // size = 228*329 + 54 + 256*4 = 76090 (228 % 4 = 0 instead of 225)
  // for 4-bits (16 colors):
  // size = 228*329 + 54 + 16*4  = 75130 (really size = 38282)
  // For the compressed file the formulas are same
  //
  //  int file_size = PathName().FileSize();
  int offset = uInt32Number(header + 10);

  // Read next 4 bytes from BITMAPINFOHEADER  it mast be 40 (0x28)
  file.Read(header, 4, numb);
  if (numb != 4)
    return false;
  if (uInt32Number(header) != 40)
    return false;

  bitmap_pos = offset;
  return true;
} 

struct LoaderColor
{
    unsigned char r, g, b, a;
};

bool gml::SimpleImageLoader::LoadBMP(const std::string& path,
                                     Image& out_bitmap)
{
  LoaderColor palette[256];

  gml::File f(path);
  // Open file
  if (!f.Open("rb"))
    return false;

  int bitmap_pos = 0;
  if (!ReadFileHeader(f, bitmap_pos))
  {
    (void) f.Close();
    return false;
  }

  // Previous function read BMP_SIZE_FILEHEADER + 4 bytes
  if (!f.Seek(BMP_SIZE_FILEHEADER))
  {
    (void) f.Close();
    return false;
  }

  // Read BITMAPINFOHEADER
  BYTE buf[40];
  int numb = 0;
  f.Read(buf, 40, numb);
  if (numb != 40)
  {
    (void) f.Close();
    return false;
  }

  int x_res = (int) uInt32Number(buf + 4);
  int y_res = (int) uInt32Number(buf + 8);
  /*
  if (xres != x_res || yres != y_res)
    {
    (void)f.Close();
    return false;
    }*/

  //           Parametes of the bitmap data
  // int n_planes        = (int)uInt16Number(buf + 12);
  int n_bits = (int) uInt16Number(buf + 14);
  int compression = (int) uInt32Number(buf + 16);
  int size_image = (int) uInt32Number(buf + 20);
  //  int x_per_meter     = (int)uInt32Number(buf + 24);
  //  int y_per_meter     = (int)uInt32Number(buf + 28);
  int n_used_colors = (int) uInt32Number(buf + 32);
  //  int n_cls_important = (int)uInt32Number(buf + 36);

  // --------------------------------------------------------------
  //                         TRUE color
  // --------------------------------------------------------------
  if (n_bits == BMP_COLOR_BITS_24)
  {
    // In this case there not table RGBQADR
    if (bitmap_pos != BMP_SIZE_FILEHEADER + BMP_SIZE_INFOHEADER)
    {
      (void) f.Close();
      return false;
    }

    if (!f.Seek(bitmap_pos))
    {
      (void) f.Close();
      return false;
    }

    int rgb_size = rgb_buffer_size(x_res, BMP_COLOR_BITS_24);

    // now create GML image
    gml::Image::REPRES r = gml::Image::R_BYTE;
    gml::Image::FORMAT fr = gml::Image::F_RGB;
    out_bitmap.Create(x_res, y_res, fr, r);
    if (!out_bitmap.Created())
      return false;

    BYTE** dst_lines = out_bitmap.GetLineArray();

    BYTE* rgb = new BYTE[rgb_size];


    for (int y = 0; y < y_res; y++)
    {
      int numb = 0;
      f.Read(rgb, rgb_size, numb);
      if (numb != rgb_size)
      {
        (void) f.Close();
        delete[] rgb;
        return false;
      }

      numb = 0;
      BYTE* line = dst_lines[y];
      for (int x = 0; x < x_res; x++)
      {
        line[2] = rgb[numb++];
        line[1] = rgb[numb++];
        line[0] = rgb[numb++];
        line += 3;
      }
    }
    f.Close();

    delete[] rgb;
  }
  else
    // --------------------------------------------------------------
    //                         palette
    // --------------------------------------------------------------
  {
    // here the palette is read
    BYTE buf2[4 * 256];
    memset(palette, 0, sizeof(palette));
    int palette_size = n_used_colors == 0 ? 1 << n_bits : n_used_colors;
    f.Read(buf2, palette_size * 4, numb) ;
    for (int i = 0; i < palette_size; i++)
    {
      palette[i].b = buf2[i * 4];
      palette[i].g = buf2[i * 4 + 1];
      palette[i].r = buf2[i * 4 + 2];
      palette[i].a = buf2[i * 4 + 3];
    }

    // create GML image of proper size
    gml::Image::REPRES r = gml::Image::R_BYTE;
    gml::Image::FORMAT fr = gml::Image::F_RGB;
    out_bitmap.Create(x_res, y_res, fr, r);
    if (!out_bitmap.Created())
      return false;

    BYTE** dst_lines = out_bitmap.GetLineArray();

    int src_pitch = ((x_res* n_bits + 7) / 8 + 3) & -4;
    const  int buffer_size = 1 << 12;
    unsigned char  buffer[buffer_size];
    unsigned char * src = buffer;
    unsigned char * src_line;

    // fill the bitmap
    if (n_bits == BMP_COLOR_BITS_08)
    {
      // image of 8 bites per line
      for (int y = 0; y < y_res; y++, dst_lines++)
      {
        f.Read(src, src_pitch, numb);
        src_line = src;
        unsigned char * cur_line = *dst_lines;
        unsigned char * end = cur_line + x_res * 3;
        while ((cur_line += 3) < end)
        {
          *((LoaderColor *) (cur_line - 3)) = palette[*src_line++];
        }
        LoaderColor clr = palette[src_line[0]];
        ((unsigned char *) (cur_line - 3))[0] = (clr).r;
        ((unsigned char *) (cur_line - 3))[1] = (clr).g;
        ((unsigned char *) (cur_line - 3))[2] = (clr).b;
      }
    }
    else if (n_bits == BMP_COLOR_BITS_04)
    {
      // image of 4 bites per line
      for (int y = 0; y < y_res; y++, dst_lines++)
      {
        f.Read(src, src_pitch, numb);
        src_line = src;
        unsigned char * cur_line = *dst_lines;
        unsigned char * end = cur_line + x_res * 3;
        while ((cur_line += 6) < end)
        {
          int idx = *src_line++;
          *((LoaderColor *) (cur_line - 6)) = palette[idx >> 4];
          *((LoaderColor *) (cur_line - 3)) = palette[idx & 15];
        }
        int idx = src_line[0];
        LoaderColor clr = palette[idx >> 4];
        ((unsigned char *) (cur_line - 6))[0] = (clr).r;
        ((unsigned char *) (cur_line - 6))[1] = (clr).g;
        ((unsigned char *) (cur_line - 6))[2] = (clr).b;
        if (cur_line == end)
        {
          clr = palette[idx & 15];
          ((unsigned char *) (cur_line - 3))[0] = (clr).r;
          ((unsigned char *) (cur_line - 3))[1] = (clr).g;
          ((unsigned char *) (cur_line - 3))[2] = (clr).b;
        }
      }
    }
    else if (n_bits == BMP_COLOR_BITS_01)
    {
      // image of 1 bite per line
      for (int y = 0; y < y_res; y++, dst_lines++)
      {
        f.Read(src, src_pitch, numb);
        src_line = src;
        unsigned char * cur_line = *dst_lines;
        unsigned char * end = cur_line + x_res * 3;
        while ((cur_line += 24) < end)
        {
          int idx = *src_line++;
          *((LoaderColor *) (cur_line - 24)) = palette[(idx & 128) != 0];
          *((LoaderColor *) (cur_line - 21)) = palette[(idx & 64) != 0];
          *((LoaderColor *) (cur_line - 18)) = palette[(idx & 32) != 0];
          *((LoaderColor *) (cur_line - 15)) = palette[(idx & 16) != 0];
          *((LoaderColor *) (cur_line - 12)) = palette[(idx & 8) != 0];
          *((LoaderColor *) (cur_line - 9)) = palette[(idx & 4) != 0];
          *((LoaderColor *) (cur_line - 6)) = palette[(idx & 2) != 0];
          *((LoaderColor *) (cur_line - 3)) = palette[(idx & 1) != 0];
        }

        int idx = src_line[0] << 24;
        for (cur_line -= 24; cur_line < end; cur_line += 3, idx += idx)
        {
          LoaderColor clr = palette[idx < 0];

          ((unsigned char *) (cur_line))[0] = (clr).r;
          ((unsigned char *) (cur_line))[1] = (clr).g;
          ((unsigned char *) (cur_line))[2] = (clr).b;
        }
        LoaderColor clr = palette[src_line[0]];
      }
    }
    else
      return false;
  }

  return true;
}


// -------------------------------------------------------
//              STATIC FUNCTION DEFINITIONS
// -------------------------------------------------------

// -------------------------------------------------------
// NAME       uInt16Number()
// PURPOSE    Convert two bytes to int 16 bits
// RETURNS    number
// -------------------------------------------------------
static unsigned int uInt16Number(BYTE buf[2])
{
  return (buf[1] << 8) | buf[0];
}
// END OF METHOD uInt16Number()

// -------------------------------------------------------
// NAME       uInt32Number()
// PURPOSE    Convert four bytes to int 32 bits
// RETURNS    number
// -------------------------------------------------------
static unsigned int uInt32Number(BYTE buf[4])
{
  unsigned numb = buf[3];
  numb = (numb << 8) | buf[2];
  numb = (numb << 8) | buf[1];
  numb = (numb << 8) | buf[0];
  return numb;
}
// END OF METHOD uInt32Number()

// -------------------------------------------------------
// NAME       IntTo2Bytes()
// PURPOSE    Convert integer to two bytes
// RETURNS    none
// -------------------------------------------------------
static void IntTo2Bytes(int val, BYTE buf[2])
{
  buf[0] = (BYTE) val;
  buf[1] = (BYTE) (val >> 8);
}
// END OF METHOD IntTo2Bytes()

// -------------------------------------------------------
// NAME       IntTo4Bytes()
// PURPOSE    Convert integer to four bytes
// RETURNS    none
// -------------------------------------------------------
static void IntTo4Bytes(int val, BYTE buf[4])
{
  buf[0] = (BYTE) val;
  buf[1] = (BYTE) (val >> 8);
  buf[2] = (BYTE) (val >> 16);
  buf[3] = (BYTE) (val >> 24);
}
// END OF METHOD IntTo4Bytes()

// -------------------------------------------------------
// NAME       rgb_buffer_size()
// PURPOSE    rgb buffer must be added to 4 
// RETURNS    new size of buffer
// -------------------------------------------------------
static int rgb_buffer_size(int xres, int type_rgb)
{
  int size = 0;
  if (type_rgb == BMP_COLOR_BITS_24)
  {
    size = 3 * xres;
  }
  else if (type_rgb == BMP_COLOR_BITS_08)
  {
    size = xres;
  }
  else if (type_rgb == BMP_COLOR_BITS_04)
  {
    size = xres / 2;
    if (size * 2 < xres)
      size++;
  }
  else if (type_rgb == BMP_COLOR_BITS_01)
  {
    size = xres / 8;
    int rest_8 = size % 8;
    if (rest_8 > 0)
      size += 8 - rest_8;
  }
  ASSERT(size > 0);

  // for the BITS_01 it is not obviously
  int rest_4 = size % 4;
  if (rest_4 > 0)
    size += 4 - rest_4;

  return size;
}
// END OF FUNCTION rgb_buffer_size(int xres);

// -------------------------------------------------------
// NAME       HalfOfByte()
// PURPOSE    Convert first/last 4 bits of byte to int
// RETURNS    integer 
// -------------------------------------------------------
static int HalfOfByte(BYTE val, int half)
{
  ASSERT(half == 0 || half == 1);
  if (half == 0)
    return (int) (val >> 4);
  return (int) (val & 15);
}
// END OF FUNCTION HalfOfByte()

// -------------------------------------------------------
// NAME       BitsOfByte()
// PURPOSE    Separation byte on bits 
// RETURNS    none
// -------------------------------------------------------
static void BitsOfByte(BYTE val, int* bits)
{
  if (((int) val & 1) == 0)
    bits[7] = 0;
  else
    bits[7] = 1;

  if (((int) val & 2) == 0)
    bits[6] = 0;
  else
    bits[6] = 1;

  if (((int) val & 4) == 0)
    bits[5] = 0;
  else
    bits[5] = 1;

  if (((int) val & 8) == 0)
    bits[4] = 0;
  else
    bits[4] = 1;

  if (((int) val & 16) == 0)
    bits[3] = 0;
  else
    bits[3] = 1;

  if (((int) val & 32) == 0)
    bits[2] = 0;
  else
    bits[2] = 1;

  if (((int) val & 64) == 0)
    bits[1] = 0;
  else
    bits[1] = 1;

  if (((int) val & 128) == 0)
    bits[0] = 0;
  else
    bits[0] = 1;
}
// END OF FUNCTION BitsOfByte() 


//////////////////////////////////////////////////////////////////////////////////
//

#ifndef __BORLANDC__
typedef struct tagBITMAPFILEHEADER
{
    WORD bfType;
    DWORD bfSize;
    WORD bfReserved1;
    WORD bfReserved2;
    DWORD bfOffBits;
} BITMAPFILEHEADER;

typedef struct tagBITMAPINFOHEADER
{
    DWORD biSize;
    long biWidth;
    long biHeight;
    WORD biPlanes;
    WORD biBitCount;
    DWORD biCompression;
    DWORD biSizeImage;
    long biXPelsPerMeter;
    long biYPelsPerMeter;
    DWORD biClrUsed;
    DWORD biClrImportant;
} BITMAPINFOHEADER;

#endif

bool gml::SimpleImageLoader::SaveBMP(const std::string& path,
                                     const Image& out_bitmap)
{
  ASSERT(out_bitmap.GetFormat() == gml::Image::F_RGB ||
         out_bitmap.GetFormat() == gml::Image::F_BGR);
  ASSERT(out_bitmap.GetRepres() == gml::Image::R_BYTE);

  gml::File out_file(path);
  if (!out_file.Open("wb"))
    return false;

  int data_size = 3 * out_bitmap.GetWidth() * out_bitmap.GetHeight();

  // File header set & write 
  BYTE file_header[BMP_SIZE_FILEHEADER];

  memset(file_header, 0, BMP_SIZE_FILEHEADER);
  file_header[0] = 'B';
  file_header[1] = 'M';
  //IntTo4Bytes(56 + data_size, &(file_header[2]));
  IntTo4Bytes(54, &(file_header[10]));

  out_file.Write(file_header, BMP_SIZE_FILEHEADER);


  // Info header set & write

  BYTE info_header[BMP_SIZE_INFOHEADER];
  memset(info_header, 0, BMP_SIZE_INFOHEADER);
  IntTo4Bytes(40, &(info_header[0]));
  IntTo4Bytes(out_bitmap.GetWidth(), &(info_header[4]));
  IntTo4Bytes(out_bitmap.GetHeight(), &(info_header[8]));
  IntTo2Bytes(1, &(info_header[12]));
  IntTo2Bytes(24, &(info_header[14]));


  out_file.Write(info_header, BMP_SIZE_INFOHEADER);

  // width should be 'kratna 4m' :)
  int iFullWdt = out_bitmap.GetWidth() * out_bitmap.Format2Channels(out_bitmap.GetFormat());
  int correct_width = (iFullWdt) % 4 != 0 ? (iFullWdt / 4 * 4 + 4) : iFullWdt;

  int width_diff = correct_width - iFullWdt;
  BYTE* data = new BYTE[correct_width* out_bitmap.GetHeight()];

  BYTE** lines = out_bitmap.GetLineArray();

  unsigned int id = 0, iYStart, iYEnd, iYStep;

  if (out_bitmap.GetOrient() == gml::Image::O_TOPLEFT)
  {
    iYStart = out_bitmap.GetHeight() - 1;
    iYEnd = -1;
    iYStep = -1;
  }
  else
  {
    iYStart = 0;
    iYEnd = out_bitmap.GetHeight();
    iYStep = 1;
  }

  switch (out_bitmap.GetFormat())
  {
    case gml::Image::F_RGB:
      {
        for (int y = iYStart; y != iYEnd; y += iYStep)
        {
          for (int x = 0; x < 3 * out_bitmap.GetWidth(); x += 3)
          {
            data[id++] = lines[y][x + 2];
            data[id++] = lines[y][x + 1];
            data[id++] = lines[y][x];
          }
          id += width_diff;
        }
        break;
      }
    case gml::Image::F_BGR:
      {
        for (int y = iYStart; y != iYEnd; y += iYStep)
        {
          for (int x = 0; x < 3 * out_bitmap.GetWidth(); x += 3)
          {
            data[id++] = lines[y][x];
            data[id++] = lines[y][x + 1];
            data[id++] = lines[y][x + 2];
          }
          id += width_diff;
        }
        break;
      }
    default:
      return false;
      break;
  }

  //memset(data, 0, out_bitmap.GetHeight() * out_bitmap.GetWidth() * 3);

  out_file.Write(data, out_bitmap.GetHeight() * correct_width);

  out_file.Flush();
  out_file.Close();

  delete[] data;

  return true;
}


