//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlsimpleimage.h,v 1.18 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#ifndef _GMLSIMPLEIMAGE_H_
#define _GMLSIMPLEIMAGE_H_

#include "gmlimage.h"

namespace gml
{
  /** @addtogroup Images
   *  @{
   */

  /// Manipulates uncompressed device- and platform-independent bitmaps.

  /// The data is stored sequentially without padding in the bitmap. The class implements exactly
  /// the interface defined by gml::Image without additions.
  class SimpleImage : public Image
  {
    public:

      /// Creates an empty bitmap.
      SimpleImage();

      /// Destroys the bitmap.
      virtual ~SimpleImage();

      /// Copy constructor
      SimpleImage(const Image& Orig);

      /// Copy constructor
      SimpleImage(const SimpleImage& Orig);

      /// Assignment operator
      SimpleImage& operator =(const Image& Orig);

      // Assignment operator.
      SimpleImage& operator =(const SimpleImage& Orig);

      // SimpleImage information

      /// Returns the amount of memory used by the object.
      long GetMemUsed();

      /// Returns number of bytes used per line.
      long GetBytesPerLine() const;

      // own interface
      const BYTE* GetRawData() const
      {
        return m_pBits;
      }

      /// Access to raw data array. Dangerous function! Use with care.
      BYTE* GetRawData()
      {
        return m_pBits;
      } 

      virtual int GetRawDataSize() const
      {
        return GetBitsMemNeeded(GetWidth(),
                                GetHeight(),
                                GetFormat(),
                                GetRepres());
      }

      // Static functions

      /// Returns memory needed by a bitmap with the specified attributes.
      static long GetMemNeeded(int width, int height, FORMAT f, REPRES r);

      /// Returns memory needed by bitmap bits.
      static long GetBitsMemNeeded(int width, int height, FORMAT f, REPRES r);

      /// Draw surface
      virtual gml::DrawSurface* GetDrawSurface();

    protected:
      // Local functions

      virtual void ConstructorInitLocals();

      virtual bool InternalCreate(int Width,
                                  int Height,
                                  FORMAT format,
                                  REPRES repres,
                                  ORIENT orient = O_BOTTOMLEFT);

      // Deletes memory allocated by member variables.
      virtual void FreeMembers();

      // Initializes internal table of line addresses.
      virtual void InitLineArray();

      virtual void InternalChangeOrientation(ORIENT new_orient);

      BYTE* m_pBits;        ///< Pointer to the bits.
  };

  // Note that _both_ these copy constructors are needed. If only the 
  // second one is there, MSVC generates a default copy 
  // constructor anyway :-(.
  inline SimpleImage::SimpleImage(const SimpleImage& Orig) : Image()
  {
    ConstructorInitLocals();
    InternalCopy(Orig);
  }

  inline SimpleImage::SimpleImage(const Image& Orig) : Image()
  {
    ConstructorInitLocals();
    InternalCopy(Orig);
  }

  inline SimpleImage& SimpleImage::operator=(const Image& Orig)
  {
    Image::operator = (Orig);
    return *this;
  }

  inline SimpleImage& SimpleImage::operator=(const SimpleImage& Orig)
  {
    Image::operator = (Orig);
    return *this;
  }

  /** @} */
} // namespace gml

#endif
