//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlintelimage.cpp,v 1.8 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#ifdef GML_USE_MFC
#include <afxwin.h>
#endif
#include "../../base/gmlcommon.h"
#include "gmlintelImage.h"
#include "gmlIntelSurface.h"
#include "filters/gmlfilterrgb2bgr.h"
#include <memory.h>
#include <ipl.h>

using gml::Image;
using gml::IntelImage;
using gml::ImageFilter;



//===================================================================
//= Function name : IntelImage::IntelImage
//= Description   : 
//= Return type   : 
//===================================================================

IntelImage::IntelImage() : m_pBits(NULL)
    // Creates an empty bitmap.

{
  ConstructorInitLocals();
  Create(1, 1, F_RGB, R_BYTE);
}

//===================================================================
//= Function name : IntelImage::~IntelImage
//= Description   : 
//= Return type   : 
//===================================================================

IntelImage::~IntelImage()
{
  // Free the memory.
  FreeMembers();
}


void IntelImage::ConstructorInitLocals()
{
  m_pImage = NULL;
  m_pBits = NULL;
}

//===================================================================
//= Function name : IntelImage::GetMemUsed
//= Description   : 
//= Return type   : long 
//===================================================================

long IntelImage::GetMemUsed()
  // Returns the memory used by the object.
{
  return m_pImage->nSize +
         sizeof(*this) +
         m_pImage->imageSize +
         sizeof(BYTE *) * m_height;
}

//===================================================================
//= Function name : IntelImage::GetBytesPerLine
//= Description   : 
//= Return type   : long 
//===================================================================

long IntelImage::GetBytesPerLine() const 
  // Returns number of bytes used per line.
{
  return m_pImage->widthStep;
}

/////////////////////////////////////////////////////////////////////
// Local functions


//===================================================================
//= Function name : IntelImage::InternalCreate
//= Description   : 
//= Return type   : void 
//===================================================================

bool IntelImage::InternalCreate(int Width,
                                int Height,
                                FORMAT format,
                                REPRES repres,
                                ORIENT orient)
  // Create a new empty bitmap. Bits are uninitialized.
  // Assumes that no memory is allocated before the call.
{
  int iChannels, iBitDepth, iOrigin, iAlpha;
  ASSERT(Width > 0 && Height > 0);
  if (Width < 1 || Height < 1)
    return false;

  iChannels = Format2Channels(format);
  iBitDepth = Repres2Size(repres); 

  switch (orient)
  {
    case O_TOPLEFT:
      iOrigin = IPL_ORIGIN_TL;
      break;
    case O_BOTTOMLEFT:
      iOrigin = IPL_ORIGIN_BL;
      break;
    default:
      return false;
  }

  switch (format)
  {
    case F_L:
      iAlpha = 0;
      break;
    case F_RGB:
    case F_BGR:
      iAlpha = 0;
      break;
    case F_RGBA:
      iAlpha = 3;
      break;
    default:
      return false;
  }

  m_pImage = iplCreateImageHeader(iChannels,
                                  iAlpha,
                                  iBitDepth << 3,
                                  "GML",
                                  "GML",
                                  IPL_DATA_ORDER_PIXEL,
                                  iOrigin,
                                  IPL_ALIGN_DWORD,
                                  Width,
                                  Height,
                                  NULL,
                                  NULL,
                                  NULL,
                                  NULL);

  if (!m_pImage)
    return false;

  if (iBitDepth >= 4)
    iplAllocateImageFP(m_pImage, 0, 0);
  else
    iplAllocateImage(m_pImage, 0, 0);

  m_pBits = (unsigned char *) m_pImage->imageData;

  InitLocals(Width, Height, format, repres, orient);
  return true;
}



//===================================================================
//= Function name : IntelImage::InitLineArray
//= Description   : 
//= Return type   : void 
//===================================================================

void IntelImage::InitLineArray()
{
  m_line_array = new BYTE * [m_height];
  int LineLen = GetBytesPerLine();

  for (int y = 0; y < m_height; y++)
  {
    m_line_array[y] = m_pBits + y * LineLen;
  }
}


//===================================================================
//= Function name : IntelImage::FreeMembers
//= Description   : 
//= Return type   : void 
//===================================================================

void IntelImage::FreeMembers()
{
  if (m_pImage)
  {
    iplDeallocate(m_pImage, IPL_IMAGE_ALL);
    m_pImage = NULL;
    m_pBits = NULL;
  }

  delete[] m_line_array;
  m_line_array = NULL;
}



//===================================================================
//= Function name : IntelImage::InternalChangeOrientation
//= Description   : 
//= Return type   : void 
//===================================================================

void IntelImage::InternalChangeOrientation(ORIENT new_orient)
{
  if (m_orient == new_orient)
    return; // change nothing

  // since we've got only two orientations, the code is the same for changing one to another
  // FIXME is this OK?
  for (int y = 0; y < m_height / 2; ++y)
  {
    BYTE* line1 = m_line_array[y];
    BYTE* line2 = m_line_array[m_height - y - 1];
    for (int x = 0; x < GetBytesPerLine() ; ++x, ++line1, ++line2)
    {
      std::swap(*line1, *line2);
    }
  }

  switch (new_orient)
  {
    case O_TOPLEFT:
      m_pImage->origin = IPL_ORIGIN_TL;
      break;
    case O_BOTTOMLEFT:
      m_pImage->origin = IPL_ORIGIN_BL;
      break;
  }

  m_orient = new_orient;
}

//===================================================================
//= Function name : IntelImage::GetDrawSurface
//= Description   : 
//= Return type   : void 
//===================================================================

gml::DrawSurface* IntelImage::GetDrawSurface()
{
  return new gml::IntelSurface(this);
}

