//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlimage.h,v 1.35 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#ifndef _GMLIMAGE_H_
#define _GMLIMAGE_H_

#ifdef GML_USE_MFC
#include "afxwin.h"
class CDC;
class CPoint;
class CSize;
#endif

#include "../../base/gmlsmartobject.h"
#include "../../base/gmlcommon.h"

#include <vector>
struct tagBITMAPINFO;

namespace gml
{
  /** @addtogroup Images
   *  @{
   */


  class ImageFilter;
  class ImageDecomposer;
  class ImageComposer;
  class DrawSurface;

  /// Generic image class
  /** Some functionality should be defined in children */
  class Image : public SmartObject
  {
    public:

      enum FORMAT { F_L, F_RGB, F_RGBA, F_BGR };
      enum REPRES { R_BYTE, R_WORD, R_DWORD, R_FLOAT, R_DOUBLE };

      /// line order in the image 
      enum ORIENT { O_TOPLEFT,  ///< lines are going left-right, top-bottom
      O_BOTTOMLEFT ///< lines are left-right, bottom-top
      }; 

      static inline int Format2Channels(FORMAT f)
      {
        switch (f)
        {
          case F_L:
            return 1;
          case F_RGB:
            return 3;
          case F_BGR:
            return 3;
          case F_RGBA :
            return 4;
          default:
            ASSERT(false);
        }
        return -1;
      }

      static inline int Repres2Size(REPRES r)
      {
        switch (r)
        {
          case R_BYTE:
            return sizeof(BYTE);
          case R_WORD:
            return sizeof(WORD);
          case R_DWORD:
            return sizeof(int);
          case R_FLOAT:
            return sizeof(float);
          case R_DOUBLE:
            return sizeof(double);
          default:
            ASSERT(false);
        }
        return -1;
      }

      static inline int ElemSize(FORMAT f, REPRES r)
      {
        return Repres2Size(r) * Format2Channels(f);
      }

    public:

      /// Empty constructor. Constructors in derived classes create a
      /// small empty bitmap to ensure that the object is always in a
      /// sane state.
      Image();

      /// Empty destructor.
      virtual ~Image();


      // copy constructor 
      // by Helga von Bulow 
      Image(Image& in_Image);


      /// Assignment operator. Note that assignment between different derived
      /// classes is possible and results in a format conversion.
      Image& operator =(Image const & Orig);

      /// Test for equality. This function actually tests every pixel, so
      /// it's not fast. It's meant mainly for use in asserts and such.
      bool const operator ==(Image const & Other);

      // /// @name Image creation

      /// Creates a new empty bitmap. Memory for the bits is allocated
      /// but not initialized. Previous contents of the bitmap object are
      /// discarded
      virtual void Create(int Width,
                          int Height,
                          FORMAT format,
                          REPRES repres,
                          ORIENT orient = O_BOTTOMLEFT);

      /// Creates a copy of rSrImage, converting color depth if nessesary.
      void CreateCopy(const Image& rSrImage);

      /// Creates from a bitmap header
      /**
          @param in_pDIB   BITMAPINFO (  DIB')
          @param in_pcData    bitmap .    
                 NULL - ,      BITMAPINFO
      */
      bool CreateFromDIB(struct tagBITMAPINFO* in_pDIB, BYTE* in_pcData = NULL);

      /// Creates a copy of rSrImage, applying rFilter on the way. Depending
      /// on the filter called, this is often much faster than CreateCopy()
      /// followed by ApplyFilter().
      void CreateFilteredCopy(const Image& rSrImage,
                              const ImageFilter& rFilter);
      void CreateComposition(const std::vector<Image*>& src,
                             const ImageComposer& comp);

      bool Created() const;
      void Clear();

      // /// @name Image manipulation

      /// Applies a filter to the bitmap.
      void ApplyFilter(const ImageFilter& filter);

      /// Applies a decompositor to the bitmap.
      void Decompose(const ImageDecomposer& decomposer,
                     std::vector<Image*>& out_res);

      /// @brief Change orientation of the image to new one
      /// @note if it already set, nothing will changed
      bool ChangeOrientation(ORIENT new_orient);

      /// @brief Change image format
      /// @note if it already set, nothing will changed, currently implemented only for RGB <-> BGR change

      bool ChangeFormat(FORMAT new_format);

      /// @brief Change image pixel representetion
      /// @note if it already set, nothing will changed

      bool ChangeRepres(REPRES new_repres);

      // /// @name Image information.

      inline int GetWidth() const;

      inline int GetHeight() const;

      inline FORMAT GetFormat() const;

      inline REPRES GetRepres() const;

      inline ORIENT GetOrient() const;

      virtual long GetMemUsed() = 0;

      inline int GetElemSize() const;

      /// Returns number of bytes used per line.
      virtual long GetBytesPerLine() const = 0;

      // /// @name Image direct manipulation

      /// Returns pointer to an array containing the starting addresses of the bitmap lines. 
      /** This array should be used whenever the bitmap
        * bits need to be manipulated directly.
        */
      BYTE** GetLineArray() const;

      /// Locks bitmap. GetLineArray() and other direct-access methods should
      /// only be called if the bitmap is locked. Lock and Unlock keep a lock
      /// count.
      virtual void Lock(bool bReadable, bool bWriteable);

      /// Unlocks the Bitmap surface.
      virtual void Unlock();

      bool IsLocked() const;

      bool AlmostEqual(const Image& Bmp, int epsilon) const;

#ifdef GML_USE_MFC
      void PaintImage(CDC* in_pDC,
                      CPoint dst_org,
                      CSize dst_size,
                      DWORD dwRop = SRCCOPY);
#endif

      // /// @name Drawing on image 
      virtual gml::DrawSurface* GetDrawSurface() = 0;

    protected:


      /// Create a new bitmap with uninitialized bits. (Assume no memory
      /// is allocated yet.)
      virtual bool InternalCreate(int Width,
                                  int Height,
                                  FORMAT format,
                                  REPRES repres,
                                  ORIENT orient = O_BOTTOMLEFT) = 0;

      /// Delete memory allocated by member variables.
      virtual void FreeMembers() = 0;

      /// Initialize internal table of line addresses.
      virtual void InitLineArray() = 0;

      /// Creates a new Image as copy of rSrImage. Assumes there is no memory
      /// allocated yet.
      void InternalCopy(const Image& rSrImage);

      /// Can be called from internalCreate() to initialize object state.
      void InitLocals(int width,
                      int height,
                      FORMAT format,
                      REPRES repres,
                      ORIENT orient = O_BOTTOMLEFT);

      /// Change image orientation to given state
      /// @warning you should set m_orient variable after actual operation
      virtual void InternalChangeOrientation(ORIENT new_orient) = 0;

    protected:

      // Member variables

      int m_width;
      int m_height;
      bool m_created;
      FORMAT m_format;
      REPRES m_repres;
      ORIENT m_orient; ///< current orientation order

      BYTE** m_line_array;   ///< Table of the starting addresses of the lines.
      int m_lock_count;   ///< Number of times the bitmap was locked.

      ///<Default is m_LockCount always >= 1, so
      ///< access to bits is always possible.
  };

  inline Image& Image::operator=(Image const & Orig)
  {
    if (this != &Orig)
      CreateCopy(Orig);
    return *this;
  }

  // Image information

  inline int Image::GetWidth() const
  {
    return m_width;
  }


  inline int Image::GetHeight() const
  {
    return m_height;
  }

  inline Image::FORMAT Image::GetFormat() const
  {
    return m_format;
  }

  inline Image::REPRES Image::GetRepres() const
  {
    return m_repres;
  }

  inline Image::ORIENT Image::GetOrient() const
  {
    return m_orient;
  }

  inline int Image::GetElemSize() const
  {
    return ElemSize(m_format, m_repres);
  }


  // Image direct manipulation

  inline BYTE** Image::GetLineArray() const
  {
    return m_line_array;
  }

  inline bool Image::IsLocked() const
  {
    return (m_lock_count >= 1);
  }

  /** @} */
} // namespace gml

#endif
