//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlimage.cpp,v 1.35 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#ifdef GML_USE_MFC
#include <afxwin.h>
#endif
#include "../../base/gmlcommon.h"
#include <windows.h>
#include "../../math/gmlmath.h"
#include "gmlimage.h"
#include "gmlimagefilter.h"
#include "gmlimagecomposer.h"
#include "filters/gmlfilterrgb2bgr.h"
#include "filters/gmlfilterrepres.h"
#include <memory.h>

using gml::Image;
using gml::ImageFilter;
using gml::ImageComposer;
using gml::ImageDecomposer;

#define MAKEFOURCC_LOCAL(ch0, ch1, ch2, ch3)                              \
    ((DWORD)(BYTE)(ch0) | ((DWORD)(BYTE)(ch1) << 8) |   \
    ((DWORD)(BYTE)(ch2) << 16) | ((DWORD)(BYTE)(ch3) << 24 ))

//////////////////////////////////////////////////////////////////////////////////////////////////
//
//  conversion YUV->RGB
//
//////////////////////////////////////////////////////////////////////////////////////////////////

#define TWIST_BITS 12
#define TAP(x) ((int)((x)*(1<<TWIST_BITS) + 0.5 ))

static const int YUV2RGB[] =
{
  TAP(1.1644),
  TAP(1.596),
  TAP(0),
  -TAP(222.9184),
  TAP(1.1644),
  -TAP(0.815),
  -TAP(0.39),
  TAP(135.6096),
  TAP(1.1644),
  TAP(0),
  TAP(2.016),
  -TAP(276.6784)
};

#define PROCESS_UV( uv_ofs, r0, g0, b0 )                                       \
    b0 = U[uv_ofs]*YUV2RGB[1+4*0] +           /* 0 + */        YUV2RGB[3+4*0]; \
    g0 = U[uv_ofs]*YUV2RGB[1+4*1] + V[uv_ofs]*YUV2RGB[2+4*1] + YUV2RGB[3+4*1]; \
    r0 =      /*  0 + */            V[uv_ofs]*YUV2RGB[2+4*2] + YUV2RGB[3+4*2];

#define PROCESS_Y( y_ofs, r, g, b, r0, g0, b0 ) \
    r = Y[y_ofs]*YUV2RGB[0+4*0];                \
    b = (r + b0) >> TWIST_BITS;                 \
    g = (r + g0) >> TWIST_BITS;                 \
    r = (r + r0) >> TWIST_BITS;

#define SATURATE(x) (BYTE)(!((x)&~255)?(x):~((x)>>31))

#define SAVE_RGB(ofs, r, g, b, pLine)     \
    pLine[(ofs)]   = SATURATE(b); \
    pLine[(ofs)+1] = SATURATE(g); \
    pLine[(ofs)+2] = SATURATE(r);


static void ConvertYUV420_TO_BGR(int width,
                                 int height,
                                 BYTE* Y,
                                 BYTE* U,
                                 BYTE* V,
                                 BYTE** in_pLineArray)
{
  int i, j;
  assert(((width | height) & 7) == 0);

  width /= 2;

  BYTE* pCurLine, * pNextLine;

  for (i = 0; i < height; i += 2, Y += 4 * width,
                                                  U += width, V += width)
  {
    pCurLine = in_pLineArray[height - 1 - i];
    pNextLine = in_pLineArray[height - 1 - i - 1];

    for (j = 0; j < width; j++, pCurLine += 6, pNextLine += 6)
    {
      int r0, g0, b0, r, g, b;

      PROCESS_UV(j, r0, g0, b0);

      PROCESS_Y(j * 2, r, g, b, r0, g0, b0);
      SAVE_RGB(0, r, g, b, pCurLine);

      PROCESS_Y(j * 2 + 1, r, g, b, r0, g0, b0);
      SAVE_RGB(3, r, g, b, pCurLine);

      PROCESS_Y(j * 2 + width * 2, r, g, b, r0, g0, b0);
      SAVE_RGB(0, r, g, b, pNextLine);

      PROCESS_Y(j * 2 + width * 2 + 1, r, g, b, r0, g0, b0);
      SAVE_RGB(3, r, g, b, pNextLine);
    }
  }
}

//////////////////////////////////////////////////////////////////////////////////////////////////
//
//  Image class start here
//
//////////////////////////////////////////////////////////////////////////////////////////////////

Image::Image() : m_lock_count(0), m_created(false), m_line_array(NULL)
{
#if (!defined __BORLANDC__) || (__BORLANDC__ >= 0x550)
  ASSERT(sizeof(m_format) == sizeof(int));
#endif
}


Image::~Image()
{
  ASSERT(m_lock_count == 0);
}

Image::Image(Image& in_Image)
{
  this->CreateCopy(in_Image);
}

bool const Image::operator ==(Image const & Other)
{
  return AlmostEqual(Other, 0);
}


void Image::CreateCopy(const Image& rSrcBmp)
{
  if (&rSrcBmp != this && rSrcBmp.Created())
  {
    // Create empty bitmap.
    InternalCopy(rSrcBmp);
    m_created = rSrcBmp.m_created;
  }
}

void Image::CreateFilteredCopy(const Image& rSrcBmp,
                               const ImageFilter& rFilter)
{
  rFilter.Apply(&rSrcBmp, this);
  m_created = true;
}

void Image::CreateComposition(const std::vector<Image*>& src,
                              const ImageComposer& comp)
{
  comp.Apply(src, this);
}

/////////////////////////////////////////////////////////////////////
// Image creation

void Image::Create(int width,
                   int height,
                   FORMAT format,
                   REPRES repres,
                   ORIENT orient)
{
  if (Created() &&
      width == GetWidth() &&
      height == GetHeight() &&
      format == GetFormat() &&
      repres == GetRepres() &&
      orient == GetOrient())
    return;

  if (Created())
    FreeMembers();
  m_created = InternalCreate(width, height, format, repres, orient);
#ifndef __BORLANDC__
  ASSERT(m_created);
#endif
}


bool Image::Created() const
{
  return m_created;
}

void Image::Clear()
{
  if (Created())
  {
    FreeMembers();
    m_created = false;
  }
}

/////////////////////////////////////////////////////////////////////
// Image manipulation

void Image::ApplyFilter(const ImageFilter& Filter)
{
  Filter.ApplyInPlace(this);
}

void Image::Decompose(const ImageDecomposer& decomposer,
                      std::vector<Image*>& out_res)
{
  decomposer.Apply(this, out_res);
}

void Image::Lock(bool bReadable, bool bWriteable)
{
  ASSERT(m_lock_count >= 0);
  ASSERT(bReadable || bWriteable);
  m_lock_count++;
}

void Image::Unlock()
{
  m_lock_count--;
  ASSERT(m_lock_count >= 0);
}


bool Image::AlmostEqual(const Image& Bmp, int epsilon) const
{
  /*
  return true; // FIXME
  if (GetWidth() != Bmp.GetWidth() ||
  GetHeight() != Bmp.GetHeight() ||
  HasAlpha() != Bmp.HasAlpha() ||
  GetBitsPerPixel() != Bmp.GetBitsPerPixel())
  return false;
    if (m_Resolution != Bmp.GetResolution())
    return false;
    
      PLBYTE ** ppLines1 = GetLineArray();
      PLBYTE ** ppLines2 = Bmp.GetLineArray();
      int y,x;
      for (y=0; y<GetHeight(); y++)
      for (x=0; x<GetWidth(); x++)
      switch (GetBitsPerPixel())
      {
      case 8:
      if (abs (ppLines1[y][x] - ppLines2[y][x]) > epsilon)
      return false;
      break;
      case 24:
      if (ppLines1[y][x*3+PL_RGBA_RED] != ppLines2[y][x*3+PL_RGBA_RED] ||
      ppLines1[y][x*3+PL_RGBA_GREEN] != ppLines2[y][x*3+PL_RGBA_GREEN] ||
      ppLines1[y][x*3+PL_RGBA_BLUE] != ppLines2[y][x*3+PL_RGBA_BLUE])
      return false;
      break;
      case 32:
      if (abs (ppLines1[y][x*4+PL_RGBA_RED] - ppLines2[y][x*4+PL_RGBA_RED]) > epsilon ||
      abs (ppLines1[y][x*4+PL_RGBA_GREEN] - ppLines2[y][x*4+PL_RGBA_GREEN]) > epsilon ||
      abs (ppLines1[y][x*4+PL_RGBA_BLUE] - ppLines2[y][x*4+PL_RGBA_BLUE]) > epsilon)
      return false;
      if (HasAlpha() &&
      abs (ppLines1[y][x*4+3] - ppLines2[y][x*4+3]) > epsilon)
      return false;
      break;
      default:
      // Unsupported BPP.
      PLASSERT (false);
      }
      
        // Test if the palettes are the same for paletted bitmaps.
        if (GetBitsPerPixel() == 8)
        {
        int i;
        PLPixel32 * pPal1 = GetPalette();
        PLPixel32 * pPal2 = Bmp.GetPalette();
        for (i=0; i<255; i++)
        {
        if (abs (pPal1[i].GetR() - pPal2[i].GetR()) > epsilon ||
        abs (pPal1[i].GetG() - pPal2[i].GetG()) > epsilon ||
        abs (pPal1[i].GetB() - pPal2[i].GetB()) > epsilon)
        return false;
        }
        }
        */
  return true;
}



/////////////////////////////////////////////////////////////////////
// Local functions


void Image::InitLocals(int width,
                       int height,
                       FORMAT format,
                       REPRES repres,
                       ORIENT orient)
{
  m_width = width;
  m_height = height;
  m_format = format;
  m_repres = repres;
  m_orient = orient;

  // Initialize pointers to lines.
  InitLineArray();
}

void Image::InternalCopy(const Image& rSrcBmp)
{
  // Create empty bitmap.
  bool bCompatible;
  if (Created() &&
      rSrcBmp.GetWidth() == GetWidth() &&
      rSrcBmp.GetHeight() == GetHeight() &&
      rSrcBmp.GetFormat() == GetFormat() &&
      rSrcBmp.GetRepres() == GetRepres() &&
      rSrcBmp.GetOrient() == GetOrient())
    bCompatible = true;
  else
  {
    FreeMembers();
    bCompatible = InternalCreate(rSrcBmp.GetWidth(),
                                 rSrcBmp.GetHeight(),
                                 rSrcBmp.GetFormat(),
                                 rSrcBmp.GetRepres(),
                                 rSrcBmp.GetOrient());
  };

  if (!bCompatible)
  {
    ASSERT(bCompatible);
    m_created = false;
    return;
  }

  // We'll change rSrcBmp back to the original state before returning,
  // so the const_cast is ok. I think.
  const_cast<Image&>(rSrcBmp).Lock(true, false);
  Lock(false, true);
  BYTE** pSrcLines = rSrcBmp.GetLineArray();
  BYTE** pDstLines = GetLineArray();

  // Minimum possible line length of both images NB!
  int LineLen = GetWidth() * GetElemSize();

  for (int y = 0; y < GetHeight(); y++)
  {
    memcpy(pDstLines[y], pSrcLines[y], LineLen);
  }

  m_created = rSrcBmp.m_created;
  Unlock();
  const_cast<Image&>(rSrcBmp).Unlock();
}

bool Image::ChangeOrientation(ORIENT new_orient)
{
  if (m_orient != new_orient)
    InternalChangeOrientation(new_orient);

#ifndef __BORLANDC__
  ASSERT(m_orient == new_orient);
#endif

  return (m_orient == new_orient);
}

bool Image::ChangeFormat(FORMAT new_format)
{
  if (m_format != new_format)
  {
    if (m_format != F_RGB &&
        m_format != F_BGR ||
        new_format != F_RGB &&
        new_format != F_BGR)
    {
      ASSERT("Not supported" == 0);
      return false;
    }

    ApplyFilter(gml::FilterRGB2BGR());
    m_format = new_format;
  }
#ifndef __BORLANDC__
  ASSERT(m_format == new_format);
#endif
  return (m_format == new_format);
}

bool Image::ChangeRepres(REPRES new_repres)
{
  if (m_repres != new_repres)
    ApplyFilter(gml::FilterRepres(new_repres));

#ifndef __BORLANDC__
  ASSERT(m_repres == new_repres);
#endif
  return (m_repres == new_repres);
}

/// Creates 
bool Image::CreateFromDIB(struct tagBITMAPINFO* in_pDIB, BYTE* in_pcData)
{
  int iWidth, iHeight;
  ORIENT Orient;

  /// Create image
  iWidth = in_pDIB->bmiHeader.biWidth;
  iHeight = abs(in_pDIB->bmiHeader.biHeight);

  if (in_pDIB->bmiHeader.biHeight < 0)
  {
    iHeight = -in_pDIB->bmiHeader.biHeight;
    Orient = O_TOPLEFT;
  }
  else
  {
    iHeight = in_pDIB->bmiHeader.biHeight;
    Orient = O_BOTTOMLEFT;
  }

  Create(iWidth, iHeight, F_BGR, R_BYTE, Orient);

  if (!Created())
    return false; //Smth went wrong

  ///    
  if (!in_pcData)
    in_pcData = ((BYTE *) in_pDIB) + in_pDIB->bmiHeader.biSize;

  ///    
  switch (in_pDIB->bmiHeader.biCompression)
  {
    case BI_RGB:
      {
        ///  RGB
        switch (in_pDIB->bmiHeader.biBitCount)
        {
          case 24:
            {
              // True color 24 bit
              int iLineLen = (iWidth * 3 + 3) & -4;
              for (int i = 0; i < GetHeight(); i++, in_pcData += iLineLen)
                memcpy(GetLineArray()[i],
                       in_pcData,
                       GetElemSize() * GetWidth());
            }
            break;
          case 8:
            {
              // Paletted 8 bit
              /// FIXME - Warning!!! Untested!!!
              int iLineLen = (iWidth + 3) & -4;
              if (!in_pDIB->bmiHeader.biClrUsed)
                in_pcData += 255 * sizeof(RGBQUAD);
              else
                in_pcData += (in_pDIB->bmiHeader.biClrUsed - 1) * sizeof(RGBQUAD);

              for (int i = 0; i < GetHeight(); i++, in_pcData += iLineLen)
                for (int j = 0; j < GetHeight(); j++)
                {
                  GetLineArray()[i][j * 3 + 0] = in_pDIB->bmiColors[in_pcData[j]].rgbBlue;
                  GetLineArray()[i][j * 3 + 1] = in_pDIB->bmiColors[in_pcData[j]].rgbGreen;
                  GetLineArray()[i][j * 3 + 2] = in_pDIB->bmiColors[in_pcData[j]].rgbRed;
                }
            }
            break;
          default:
            ///  
            return false;
            break;
        };
      }
      break;
    case MAKEFOURCC_LOCAL('Y','V','1','2'):
    case MAKEFOURCC_LOCAL('I','4','2','0'):
      {
        /// FIXME - Warning!!! Untested!!!
        /// YUV  (  )
        BYTE* Y = in_pcData,
                  * U = Y + iWidth* iHeight,
                  * V = U + iWidth* iHeight / 4;

        ConvertYUV420_TO_BGR(iWidth, iHeight, Y, U, V, GetLineArray());
      }
      break;
    default:
      ///  
      return false;
      break;
  }
  return true;
}





/////////////////////////////////////////////////////////////////////
// MFC painting functions

#ifdef GML_USE_MFC

//===================================================================
//= Function name : atsDisplayImage
//= Description   : displays (8u/8s)C1/C3/AC4 image
//= Return type   : void  
//===================================================================

void Image::PaintImage(CDC* in_pDC,
                       CPoint dst_org,
                       CSize dst_size,
                       DWORD dwRop)
{
  unsigned char storage[sizeof(BITMAPINFOHEADER) + 256 * 4];
  BITMAPINFOHEADER* bmih = (BITMAPINFOHEADER*) storage;

  char* src_data = 0;
  char* dst_data = 0;
  CSize src_size;
  int channels = 0, depth = 0;
  HBITMAP hbmp = 0;
  HDC hdc = 0;

  if (!Created())
    return;

  if (GetRepres() != R_BYTE)
  {
    assert("Non-8bit images painting not implemented");
    return;
  }
  else
    depth = 8;

  switch (GetFormat())
  {
    case F_L:
      channels = 1;
      break;
    case F_RGB:
    case F_BGR:
      channels = 3;
      break;
    default:
      assert("Only 1 and 3 channel images painting implemented");
      return;
  }


  src_size.cx = GetWidth();
  src_size.cy = GetHeight();

  bmih->biSize = sizeof(*bmih);
  bmih->biWidth = src_size.cx;
  bmih->biHeight = -src_size.cy;
  bmih->biPlanes = 1;
  bmih->biBitCount = (unsigned short) (channels * depth);
  bmih->biCompression = BI_RGB;
  bmih->biSizeImage = 0;
  bmih->biXPelsPerMeter = 0;
  bmih->biYPelsPerMeter = 0;
  bmih->biClrUsed = 0;
  bmih->biClrImportant = 0;


  if (channels == 1)
  {
    int i;
    RGBQUAD* palette = (RGBQUAD*) (storage + sizeof(BITMAPINFOHEADER));

    for (i = 0; i < 256; i++)
      palette[i].rgbBlue = palette[i].rgbRed = palette[i].rgbGreen = (unsigned char)
                           i;
  }

  hdc = CreateCompatibleDC(0);
  hbmp = CreateDIBSection(hdc,
                          (BITMAPINFO *) bmih,
                          DIB_RGB_COLORS,
                          (void * *) &dst_data,
                          0,
                          0);

  int iYStart, iYEnd, iYStep;

  if (GetOrient() == O_TOPLEFT)
  {
    iYStart = 0;
    iYEnd = GetHeight();
    iYStep = 1;
  }
  else
  {
    iYEnd = -1;
    iYStart = GetHeight() - 1;
    iYStep = -1;
  }

  if (hbmp && in_pDC)
  {
    HGDIOBJ hold = SelectObject(hdc, hbmp);
    int dst_step = (src_size.cx* channels + 3) & -4;
    int y;

    /* convert source image to RGB24 */
    for (y = iYStart; y != iYEnd; y += iYStep, dst_data += dst_step)
    {
      memcpy(dst_data, GetLineArray()[y], GetWidth() * channels);
    }

    VERIFY(SetStretchBltMode(in_pDC->m_hDC, COLORONCOLOR));
    VERIFY(StretchBlt(in_pDC->m_hDC,
                      dst_org.x,
                      dst_org.y,
                      dst_size.cx,
                      dst_size.cy,
                      hdc,
                      0,
                      0,
                      src_size.cx,
                      src_size.cy,
                      dwRop));
    SelectObject(hdc, hold);
  }

  if (hbmp)
    DeleteObject(hbmp);
  if (hdc)
    DeleteDC(hdc);
}
#endif
