//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlgdiimage.h,v 1.14 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#ifndef __GMLGDIIMAGE_H__
#define __GMLGDIIMAGE_H__

#include "gmlimage.h"
#include "windows.h"

namespace gml
{
  /** @addtogroup Images
   *  @{
   */

  /// Manipulates uncompressed device- and platform-independent bitmaps.

  /// The data is stored sequentially without padding in the bitmap. The class implements exactly
  /// the interface defined by gml::Image without additions.
  class GDIImage : public Image
  {
      friend class GDISurface;

    public:

      /// Creates an empty bitmap.
      GDIImage();

      /// Destroys the bitmap.
      virtual ~GDIImage();

      /// Copy constructor
      GDIImage(const Image& Orig);

      /// Copy constructor
      GDIImage(const GDIImage& Orig);

      /// Assignment operator
      GDIImage& operator =(const Image& Orig);

      // Assignment operator.
      GDIImage& operator =(const GDIImage& Orig);

      // GDIImage information

      /// Returns the amount of memory used by the object.
      long GetMemUsed();

      /// Returns number of bytes used per line.
      inline long GetBytesPerLine() const
      {
        return m_iBytesPerLine;
      };

      // own interface
      const BYTE* GetRawData() const
      {
        return m_pBits;
      }

      /// Access to raw data array. Warning!!! Dangerous function! Use with care.
      BYTE* GetRawData()
      {
        return m_pBits;
      }

      /// Paint image
      void PaintImage(HDC in_hDC, DWORD dwRop = SRCCOPY);

      /// Paint image at a specified offset
      void PaintImage(HDC in_hDC, POINT in_ptDstOrg, DWORD dwRop = SRCCOPY);

      /// Fit image to specified rectangle
      void PaintImage(HDC in_hDC,
                      POINT in_ptDstOrg,
                      SIZE in_DstSize,
                      DWORD dwRop = SRCCOPY);

      /// Fit image fragment to specified rectangle
      void PaintImageRect(HDC in_hDC,
                          POINT in_ptDstOrg,
                          RECT in_SrcRect,
                          SIZE in_DstSize,
                          DWORD dwRop = SRCCOPY);

      // Static functions

      /// Draw surface
      virtual gml::DrawSurface* GetDrawSurface();

      /// Fast GDI image resize
      virtual bool FastResizeImage(double in_dScale, gml::GDIImage& out_ResImg);


    protected:
      // Local functions

      virtual void ConstructorInitLocals();

      virtual bool InternalCreate(int Width,
                                  int Height,
                                  FORMAT format,
                                  REPRES repres,
                                  ORIENT orient = O_BOTTOMLEFT);

      // Deletes memory allocated by member variables.
      virtual void FreeMembers();

      // Initializes internal table of line addresses.
      virtual void InitLineArray();

      virtual void InternalChangeOrientation(ORIENT new_orient);

      static long GetBytesPerLine(int width, FORMAT f, REPRES r);

      virtual void InternalPaint(HDC in_hDC,
                                 int in_iDstOrgX,
                                 int in_iDstOrgY,
                                 int in_iDstWdt,
                                 int in_iDstHgt,
                                 int in_iSrcOrgX,
                                 int in_iSrcOrgY,
                                 int in_iSrcWdt,
                                 int in_iSrcHgt,
                                 DWORD dwRop);

      BYTE* m_pBits;            ///< Pointer to the bits.
      HDC m_hDC;              ///< Internal device context
      BITMAPINFO m_BMInfo;           ///< Bitmap info structure
      RGBQUAD m_pPalette[255];    ///< Palette for 1-channel images
      RGBQUAD* m_pColorTable;
      HBITMAP m_hBMP;             ///< BITMAP handel
      HGDIOBJ m_hOldHandle;       ///< Neded for cleanup
      int m_iBytesPerLine;    ///< Bytes per line in the created DIB
  };

  // Note that _both_ these copy constructors are needed. If only the 
  // second one is there, MSVC generates a default copy 
  // constructor anyway :-(.
  inline GDIImage::GDIImage(const GDIImage& Orig) : Image()
  {
    ConstructorInitLocals();
    InternalCopy(Orig);
  }

  inline GDIImage::GDIImage(const Image& Orig) : Image()
  {
    ConstructorInitLocals();
    InternalCopy(Orig);
  }

  inline GDIImage& GDIImage::operator=(const Image& Orig)
  {
    Image::operator = (Orig);
    return *this;
  }

  inline GDIImage& GDIImage::operator=(const GDIImage& Orig)
  {
    Image::operator = (Orig);
    return *this;
  }

  /** @} */
} // namespace gml


#endif //__GMLGDIIMAGE_H__