//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlgdiimage.cpp,v 1.18 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#include "../../base/gmlcommon.h"
#include "gmlGDIImage.h"
#include "gmlGDISurface.h"
#include <memory.h>

using gml::Image;
using gml::GDIImage;
using gml::ImageFilter;



//===================================================================
//= Function name : GDIImage::GDIImage
//= Description   : 
//= Return type   : 
//===================================================================

GDIImage::GDIImage() : m_pBits(NULL)
    // Creates an empty bitmap.

{
  ConstructorInitLocals();
  Create(1, 1, F_RGB, R_BYTE);
}

void GDIImage::ConstructorInitLocals()
{
  m_pBits = NULL;
  m_hDC = NULL;
  m_hBMP = NULL;
  m_hOldHandle = NULL;

  m_BMInfo.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
  m_BMInfo.bmiHeader.biCompression = BI_RGB;
  m_BMInfo.bmiHeader.biPlanes = 1;
  m_BMInfo.bmiHeader.biSizeImage = 0;
  m_BMInfo.bmiHeader.biXPelsPerMeter = 0;
  m_BMInfo.bmiHeader.biYPelsPerMeter = 0;
  m_BMInfo.bmiHeader.biClrImportant = 0;
  m_pColorTable = &m_BMInfo.bmiColors[0];
  m_iBytesPerLine = 0;
}
/*
// copy constructor 
// by Helga von Bulow 
GDIImage::GDIImage(Image &in_Image)
{
  m_BMInfo.bmiHeader.biCompression = BI_RGB;
  m_BMInfo.bmiHeader.biPlanes = 1;
  m_BMInfo.bmiHeader.biSizeImage = 0;
  m_BMInfo.bmiHeader.biXPelsPerMeter = 0;
  m_BMInfo.bmiHeader.biYPelsPerMeter = 0;
  m_BMInfo.bmiHeader.biClrImportant = 0;
  m_pColorTable = &m_BMInfo.bmiColors[0];
  m_iBytesPerLine = 0;
}*/

//===================================================================
//= Function name : GDIImage::~GDIImage
//= Description   : 
//= Return type   : 
//===================================================================

GDIImage::~GDIImage()
{
  // Free the memory.
  FreeMembers();
}



//===================================================================
//= Function name : GDIImage::GetMemUsed
//= Description   : 
//= Return type   : long 
//===================================================================

long GDIImage::GetMemUsed()
  // Returns the memory used by the object.
{
  return sizeof(GDIImage) +
         m_iBytesPerLine * m_height +
         sizeof(BYTE *) * m_height;
}

/////////////////////////////////////////////////////////////////////
// Static functions

//===================================================================
//= Function name : GetBytesPerLine 
//= Description   : 
//= Return type   : static long 
//===================================================================

long GDIImage::GetBytesPerLine(int width, FORMAT f, REPRES r)
{
  int nBytes = 0; 

  ASSERT(r == R_BYTE);

  switch (f)
  {
    case F_L:
      nBytes = (width + 3) & -4; 
      break;
    case F_RGB:
    case F_BGR:
      nBytes = (width * 3 + 3) & -4;
      break;
    default:
      ASSERT("Not supported format" == 0);
  }

  return nBytes;
}


/////////////////////////////////////////////////////////////////////
// Local functions


//===================================================================
//= Function name : GDIImage::InternalCreate
//= Description   : 
//= Return type   : void 
//===================================================================

bool GDIImage::InternalCreate(int Width,
                              int Height,
                              FORMAT format,
                              REPRES repres,
                              ORIENT orient)
  // Create a new empty bitmap. Bits are uninitialized.
  // Assumes that no memory is allocated before the call.
{
  ASSERT(Width > 0 && Height > 0);

  if (Width < 1 || Height < 1)
    return false;

  switch (format)
  {
    case F_L:
      {
        m_BMInfo.bmiHeader.biBitCount = 8;
        m_BMInfo.bmiHeader.biClrUsed = 256;
        // prepare palette
        for (int i = 0; i < 256; i++)
          m_pColorTable[i].rgbBlue = m_pColorTable[i].rgbRed = m_pColorTable[i].rgbGreen = (unsigned char)
                                     i;
      }
      break;
    case F_RGB:
    case F_BGR:
      {
        m_BMInfo.bmiHeader.biBitCount = 24;
        m_BMInfo.bmiHeader.biClrUsed = 0;
        // TODO - What's the difference from BGR to RGB?
      }
      break;
    default:
      // Not supported
      return false;
  }

  m_BMInfo.bmiHeader.biWidth = Width;

  switch (orient)
  {
    case O_TOPLEFT:
      // Height must be negative
      m_BMInfo.bmiHeader.biHeight = -Height;
      break;
    case O_BOTTOMLEFT:
      m_BMInfo.bmiHeader.biHeight = Height;
      break;
    default:
      return false;
  }

  if (repres != R_BYTE)
    return false; //Not supported

  m_hDC = CreateCompatibleDC(NULL);
  m_hBMP = CreateDIBSection(m_hDC,
                            &m_BMInfo,
                            DIB_RGB_COLORS,
                            (void * *) &m_pBits,
                            NULL,
                            0);
  if (!m_hBMP)
    return false;

  m_hOldHandle = SelectObject(m_hDC, m_hBMP);

  InitLocals(Width, Height, format, repres, orient);
  return true;
}



//===================================================================
//= Function name : GDIImage::InitLineArray
//= Description   : 
//= Return type   : void 
//===================================================================

void GDIImage::InitLineArray()
{
  m_line_array = new BYTE * [m_height];
  m_iBytesPerLine = GetBytesPerLine(GetWidth(), m_format, m_repres);

  for (int y = 0; y < m_height; y++)
  {
    m_line_array[y] = m_pBits + y * m_iBytesPerLine;
  }
}


//===================================================================
//= Function name : GDIImage::FreeMembers
//= Description   : 
//= Return type   : void 
//===================================================================

void GDIImage::FreeMembers()
{
  SelectObject(m_hDC, m_hOldHandle);
  DeleteObject(m_hBMP);
  DeleteDC(m_hDC);

  m_hDC = NULL;
  m_hBMP = NULL;
  m_pBits = NULL;

  delete[] m_line_array;
  m_line_array = NULL;
}



//===================================================================
//= Function name : GDIImage::InternalChangeOrientation
//= Description   : 
//= Return type   : void 
//===================================================================

void GDIImage::InternalChangeOrientation(ORIENT new_orient)
{
  if (m_orient == new_orient)
    return; // change nothing

  // Change bitmap properties
  switch (new_orient)
  {
    case O_TOPLEFT:
      // Height must be negative
      m_BMInfo.bmiHeader.biHeight = -GetHeight();
      break;
    case O_BOTTOMLEFT:
      m_BMInfo.bmiHeader.biHeight = GetHeight();
      break;
    default:
      return;
  }

  // create new windows bitmap with requested orientation
  BYTE* pBits, * pCurLine;
  HBITMAP hBMP;

  hBMP = CreateDIBSection(m_hDC,
                          &m_BMInfo,
                          DIB_RGB_COLORS,
                          (void * *) &pBits,
                          NULL,
                          0);
  if (!hBMP)
    return;

  pCurLine = pBits;

  // since we've got only two orientations, the code is the same for changing one to another  
  for (int y = m_height - 1; y >= 0; --y, pCurLine += GetBytesPerLine())
  {
    memcpy(pCurLine, m_line_array[y], sizeof(char) * GetBytesPerLine());
  }

  // Swap bitmaps and their propertiers
  VERIFY(SelectObject(m_hDC, m_hOldHandle));
  VERIFY(DeleteObject(m_hBMP));

  m_hBMP = hBMP;
  m_pBits = pBits;

  // recreate line array
  delete[] m_line_array;
  m_line_array = NULL;

  InitLineArray();
  VERIFY(m_hOldHandle = SelectObject(m_hDC, m_hBMP));

  m_orient = new_orient;
}

//===================================================================
//= Function name : GDIImage::PaintImage
//= Description   :
//= Return type   : void
//===================================================================

void GDIImage::PaintImage(HDC in_hDC, DWORD dwRop)
{
  InternalPaint(in_hDC,
                0,
                0,
                GetWidth(),
                GetHeight(),
                0,
                0,
                GetWidth(),
                GetHeight(),
                dwRop);
}

//===================================================================
//= Function name : GDIImage::PaintImage
//= Description   :
//= Return type   : void
//===================================================================

void GDIImage::PaintImage(HDC in_hDC, POINT in_ptDstOrg, DWORD dwRop)
{
  InternalPaint(in_hDC,
                in_ptDstOrg.x,
                in_ptDstOrg.y,
                GetWidth(),
                GetHeight(),
                0,
                0,
                GetWidth(),
                GetHeight(),
                dwRop);
}

//===================================================================
//= Function name : GDIImage::PaintImage
//= Description   : 
//= Return type   : void 
//===================================================================

void GDIImage::PaintImage(HDC in_hDC,
                          POINT in_ptDstOrg,
                          SIZE in_DstSize,
                          DWORD dwRop)
{
  InternalPaint(in_hDC,
                in_ptDstOrg.x,
                in_ptDstOrg.y,
                in_DstSize.cx,
                in_DstSize.cy,
                0,
                0,
                GetWidth(),
                GetHeight(),
                dwRop);
}

//===================================================================
//= Function name : GDIImage::PaintImageRect
//= Description   :
//= Return type   : void
//===================================================================

void GDIImage::PaintImageRect(HDC in_hDC,
                              POINT in_ptDstOrg,
                              RECT in_SrcRect,
                              SIZE in_DstSize,
                              DWORD dwRop)
{
  if (in_SrcRect.top < 0)
    in_SrcRect.top = 0;
  if (in_SrcRect.left < 0)
    in_SrcRect.left = 0;
  if (in_SrcRect.bottom >= GetHeight())
    in_SrcRect.bottom = GetHeight() - 1;
  if (in_SrcRect.right >= GetWidth())
    in_SrcRect.right = GetWidth() - 1;

  InternalPaint(in_hDC,
                in_ptDstOrg.x,
                in_ptDstOrg.y,
                in_DstSize.cx,
                in_DstSize.cy,
                in_SrcRect.left,
                in_SrcRect.top,
                in_SrcRect.right - in_SrcRect.left,
                in_SrcRect.bottom - in_SrcRect.top,
                dwRop);
}

//===================================================================
//= Function name : GDIImage::InternalPaint
//= Description   :
//= Return type   : void
//===================================================================

void GDIImage::InternalPaint(HDC in_hDC,
                             int in_iDstOrgX,
                             int in_iDstOrgY,
                             int in_iDstWdt,
                             int in_iDstHgt,
                             int in_iSrcOrgX,
                             int in_iSrcOrgY,
                             int in_iSrcWdt,
                             int in_iSrcHgt,
                             DWORD dwRop)
{
  if (!Created())
    return;

  if (m_hBMP && m_hDC)
  {
    VERIFY(SetStretchBltMode(in_hDC, COLORONCOLOR));
    VERIFY(StretchBlt(in_hDC,
                      in_iDstOrgX,
                      in_iDstOrgY,
                      in_iDstWdt,
                      in_iDstHgt,
                      m_hDC,
                      in_iSrcOrgX,
                      in_iSrcOrgY,
                      in_iSrcWdt,
                      in_iSrcHgt,
                      dwRop));
  }
}


//===================================================================
//= Function name : GDIImage::GetDrawSurface
//= Description   : 
//= Return type   : void 
//===================================================================

gml::DrawSurface* GDIImage::GetDrawSurface()
{
  return new gml::GDISurface(this);
}

//===================================================================
//= Function name : GDIImage::GetDrawSurface
//= Description   : 
//= Return type   : void 
//===================================================================

bool GDIImage::FastResizeImage(double in_dScale, gml::GDIImage& out_ResImg)
{
  int iWdt, iHgt;

  iWdt = GetWidth() * in_dScale;
  iHgt = GetHeight() * in_dScale;

  if (iWdt < 1)
    iWdt = 1;
  if (iHgt < 1)
    iHgt = 1;

  out_ResImg.Create(iWdt, iHgt, GetFormat(), GetRepres(), GetOrient());

  ///  resize
  POINT pPoint;
  pPoint.x = 0;
  pPoint.y = 0;

  SIZE sSize;
  sSize.cx = iWdt;
  sSize.cy = iHgt;
  PaintImage(out_ResImg.m_hDC, pPoint, sSize);

  return true;
}
