//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlsetrectfilter.cpp,v 1.7 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#include "../../../base/gmlcommon.h"
#ifdef GML_USE_MFC
#include <afxwin.h>
#endif
#include "../gmlimage.h"
#include "gmlsetrectfilter.h"
#include <vector>
#include <algorithm>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

using gml::SetRectFilter;
using gml::Image;


//===================================================================
//= Function name : SetRectFilter::SetRectFilter
//= Description   : 
//= Return type   : 
//===================================================================

SetRectFilter::SetRectFilter(double in_dValue) : ImageFilter()
{
  m_dValue = in_dValue;
};


//===================================================================
//= Function name : SetRectFilter::SetRectFilter
//= Description   : 
//= Return type   : 
//===================================================================

SetRectFilter::SetRectFilter(double in_dValue,
                             gml::Vector2i in_vMin,
                             gml::Vector2i in_vMax) : ImageFilter()
{
  SetParam(in_dValue, in_vMin, in_vMax);
};


//===================================================================
//= Function name : SetRectFilter::SetRectFilter
//= Description   : 
//= Return type   : 
//===================================================================

SetRectFilter::SetRectFilter(double in_dValue,
                             int in_iMinX,
                             int in_iMinY,
                             int in_iMaxX,
                             int in_iMaxY) : ImageFilter()
{
  SetParam(in_dValue, in_iMinX, in_iMinY, in_iMaxX, in_iMaxY);
};


//===================================================================
//= Function name : SetRectFilter::~SetRectFilter
//= Description   : 
//= Return type   : 
//===================================================================

SetRectFilter::~SetRectFilter()
{
};


//===================================================================
//= Function name : SetRectFilter::SetParam
//= Description   : 
//= Return type   : void 
//===================================================================

void SetRectFilter::SetParam(double in_dValue)
{
  m_dValue = in_dValue;
  m_Box = gml::BBox2i();
}


//===================================================================
//= Function name : SetRectFilter::SetParam
//= Description   : 
//= Return type   : void 
//===================================================================

void SetRectFilter::SetParam(double in_dValue,
                             int in_iMinX,
                             int in_iMinY,
                             int in_iMaxX,
                             int in_iMaxY)
{
  SetParam(in_dValue,
           gml::Vector2i(in_iMinX, in_iMinY),
           gml::Vector2i(in_iMaxX, in_iMaxY));
}

//===================================================================
//= Function name : SetRectFilter::SetParam
//= Description   : 
//= Return type   : void
//===================================================================

void SetRectFilter::SetParam(double in_dValue,
                             gml::Vector2i in_vMin,
                             gml::Vector2i in_vMax)
{
  m_dValue = in_dValue;
  if (in_vMax.LessOrEqual(in_vMin))
    in_vMax = in_vMin;
  m_Box.vmin = in_vMin;
  m_Box.vmax = in_vMax;
}


template <class T>
SetRect(int in_iMinX,
        int in_iMinY,
        int in_iMaxX,
        int in_iMaxY,
        T** in_pDstLines,
        int in_iChannels,
        T in_Value)
{
  if (in_Value != 0)
  {
    for (int y = in_iMinY; y <= in_iMaxY; ++y)
    {
      // For each line
      T* pDstPixel = reinterpret_cast<T*>(in_pDstLines[y]);
      for (int x = in_iMinX*in_iChannels; x <= in_iMaxX*in_iChannels; ++x)
      {
        // For each pixel
        pDstPixel[0] = in_Value;
        pDstPixel ++;
      }
    }
  }
  else
  {
    for (int y = in_iMinY; y <= in_iMaxY; ++y)
    {
      // For each line
      T* pDstPixel = reinterpret_cast<T*>(in_pDstLines[y]);  
      memset(&pDstPixel[in_iMinX],
             0,
             sizeof(T) * (in_iMaxX - in_iMinX) * in_iChannels);
    }
  }
}

template <>
SetRect(int in_iMinX,
        int in_iMinY,
        int in_iMaxX,
        int in_iMaxY,
        BYTE** in_pDstLines,
        int in_iChannels,
        BYTE in_Value)
{
  for (int y = in_iMinY; y <= in_iMaxY; ++y)
  {
    // For each line
    BYTE* pDstPixel = reinterpret_cast<BYTE*>(in_pDstLines[y]);
    memset(&pDstPixel[in_iMinX],
           in_Value,
           (in_iMaxX - in_iMinX) * in_iChannels);
  }
}


void SetRectFilter::Apply(const Image* pBmpSource, Image* pBmpDest) const
{
  *pBmpDest = *pBmpSource;
  BYTE** pDstLines = pBmpDest->GetLineArray();

  int iHeight = pBmpSource->GetHeight();
  int iWidth = pBmpSource->GetWidth();
  int iChannels = Image::Format2Channels(pBmpSource->GetFormat());

  // Validate rectangle

  gml::BBox2i Box = m_Box, ImageBox(gml::Vector2i(0, 0),
                                    gml::Vector2i(iWidth - 1, iHeight - 1));

  if (Box.NotEmpty())
  {
    Box.Intersect(ImageBox);
    if (!Box.NotEmpty())
      return; //Nothing to process
  }
  else
  {
    // Process everything
    Box = ImageBox;
  }


  switch (pBmpSource->GetRepres())
  {
    case Image::R_BYTE:
      SetRect<BYTE>(Box.vmin.x,
                    Box.vmin.y,
                    Box.vmax.x,
                    Box.vmax.y,
                    pDstLines,
                    iChannels,
                    m_dValue); 
      break;
    case Image::R_WORD:
      SetRect<WORD>(Box.vmin.x,
                    Box.vmin.y,
                    Box.vmax.x,
                    Box.vmax.y,
                    (WORD * *) pDstLines,
                    iChannels,
                    m_dValue); 
      break;
    case Image::R_FLOAT:
      SetRect<float>(Box.vmin.x,
                     Box.vmin.y,
                     Box.vmax.x,
                     Box.vmax.y,
                     (float * *) pDstLines,
                     iChannels,
                     m_dValue);
      break;
    case Image::R_DOUBLE:
      SetRect<double>(Box.vmin.x,
                      Box.vmin.y,
                      Box.vmax.x,
                      Box.vmax.y,
                      (double * *) pDstLines,
                      iChannels,
                      m_dValue);
      break;
  };
}



