//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlfilterresize.cpp,v 1.12 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#include "../../../base/gmlcommon.h"
#include "../gmlimage.h"
#include "gmlfilterresize.h"


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

using gml::FilterResize;
using gml::Image;

FilterResize::FilterResize() : ImageFilter()
{
  SetResizeParam(1, 1, FASTEST);
}

FilterResize::FilterResize(int in_iDstWdt, int in_iDstHgt, RESIZE_TYPE in_Type) : ImageFilter()
{
  SetResizeParam(in_iDstWdt, in_iDstHgt, in_Type);
}

FilterResize::~FilterResize()
{
}



//===================================================================
//= Function name : void BilinearResize
//= Description   : 
//= Return type   : template<class ElemType> 
//===================================================================

template <class ElemType>
void BilinearResize(int in_iScrWdt,
                    int in_iSrcHgt,
                    ElemType** pSrcLines,
                    int in_iDstWdt,
                    int in_iDstHgt,
                    ElemType** pDstLines,
                    int iChannels)
{
  int x, y, xP, yP, yP2, xP2;

  ElemType* pcRead, * pcRead2, * pcPixel, * pcCol1, * pcCol2;

  int t, z, z2, iz2;
  int w1, w2, w3, w4;

  if (in_iDstWdt < 1 || in_iDstHgt < 1)
    return;

  xP2 = ((in_iScrWdt - 1) << 15) / in_iDstWdt;
  yP2 = ((in_iSrcHgt - 1) << 15) / in_iDstHgt;

  yP = 0;
  for (y = 0; y <= in_iDstHgt - 1; y++)
  {
    xP = 0;
    pcRead = (ElemType *) pSrcLines[yP >> 15];// &in_pcImage[];

    if ((yP >> 16) < in_iSrcHgt - 1)
      pcRead2 = (ElemType *) pSrcLines[(yP >> 15) + 1];
    else
      pcRead2 = (ElemType *) pSrcLines[(yP >> 15)];

    pcPixel = (ElemType *) pDstLines[y];
    z2 = (yP & 0x7FFF);
    iz2 = 0x8000 - z2;

    for (x = 0; x < in_iDstWdt; x++)
    {
      t = xP >> 15;
      z = xP & 0x7FFF;
      w2 = (z * iz2) >> 15;
      w1 = iz2 - w2;
      w4 = (z * z2) >> 15;
      w3 = z2 - w4;

      pcCol1 = &pcRead[t * iChannels];
      pcCol2 = &pcRead2[t * iChannels];


      for (int i = 0; i < iChannels; i++)
        pcPixel[i] = (pcCol1[i] * w1 +
                      (pcCol1 + iChannels)[i] * w2 +
                      pcCol2[i] * w3 +
                      (pcCol2 + iChannels)[i] * w4) >>
                     15; 

      pcPixel += iChannels;

      xP += xP2;
    }
    yP += yP2;
  }
}


//===================================================================
//= Function name : BilinearResize<float>
//= Description   : 
//= Return type   : void 
//===================================================================

template <>
void BilinearResize <float>(int in_iScrWdt,
                            int in_iSrcHgt,
                            float** pSrcLines,
                            int in_iDstWdt,
                            int in_iDstHgt,
                            float** pDstLines,
                            int iChannels)
{
  int x, y, xP, yP, yP2, xP2;

  float* pcRead, * pcRead2, * pcPixel, * pcCol1, * pcCol2;

  int t, z, z2, iz2;
  int w1, w2, w3, w4;

  if (in_iDstWdt < 1 || in_iDstHgt < 1)
    return;

  xP2 = ((in_iScrWdt - 1) << 15) / in_iDstWdt;
  yP2 = ((in_iSrcHgt - 1) << 15) / in_iDstHgt;

  yP = 0;
  for (y = 0; y <= in_iDstHgt - 1; y++)
  {
    xP = 0;
    pcRead = (float *) pSrcLines[yP >> 15];// &in_pcImage[];

    if ((yP >> 16) < in_iSrcHgt - 1)
      pcRead2 = (float *) pSrcLines[(yP >> 15) + 1];
    else
      pcRead2 = (float *) pSrcLines[(yP >> 15)];

    pcPixel = (float *) pDstLines[y];
    z2 = (yP & 0x7FFF);
    iz2 = 0x8000 - z2;

    for (x = 0; x < in_iDstWdt; x++)
    {
      t = xP >> 15;
      z = xP & 0x7FFF;
      w2 = (z * iz2) >> 15;
      w1 = iz2 - w2;
      w4 = (z * z2) >> 15;
      w3 = z2 - w4;

      pcCol1 = &pcRead[t * iChannels];
      pcCol2 = &pcRead2[t * iChannels];


      for (int i = 0; i < iChannels; i++)
        pcPixel[i] = (pcCol1[i] * w1 +
                      (pcCol1 + iChannels)[i] * w2 +
                      pcCol2[i] * w3 +
                      (pcCol2 + iChannels)[i] * w4) * 0.000030517578125; 

      pcPixel += iChannels;

      xP += xP2;
    }
    yP += yP2;
  }
}

template <>
void BilinearResize <double>(int in_iScrWdt,
                             int in_iSrcHgt,
                             double** pSrcLines,
                             int in_iDstWdt,
                             int in_iDstHgt,
                             double** pDstLines,
                             int iChannels)
{
  int x, y, xP, yP, yP2, xP2;

  double* pcRead, * pcRead2, * pcPixel, * pcCol1, * pcCol2;

  int t, z, z2, iz2;
  int w1, w2, w3, w4;

  if (in_iDstWdt < 1 || in_iDstHgt < 1)
    return;

  xP2 = ((in_iScrWdt - 1) << 15) / in_iDstWdt;
  yP2 = ((in_iSrcHgt - 1) << 15) / in_iDstHgt;

  yP = 0;
  for (y = 0; y <= in_iDstHgt - 1; y++)
  {
    xP = 0;
    pcRead = (double *) pSrcLines[yP >> 15];// &in_pcImage[];

    if ((yP >> 16) < in_iSrcHgt - 1)
      pcRead2 = (double *) pSrcLines[(yP >> 15) + 1];
    else
      pcRead2 = (double *) pSrcLines[(yP >> 15)];

    pcPixel = (double *) pDstLines[y];
    z2 = (yP & 0x7FFF);
    iz2 = 0x8000 - z2;

    for (x = 0; x < in_iDstWdt; x++)
    {
      t = xP >> 15;
      z = xP & 0x7FFF;
      w2 = (z * iz2) >> 15;
      w1 = iz2 - w2;
      w4 = (z * z2) >> 15;
      w3 = z2 - w4;

      pcCol1 = &pcRead[t * iChannels];
      pcCol2 = &pcRead2[t * iChannels];


      for (int i = 0; i < iChannels; i++)
        pcPixel[i] = (pcCol1[i] * w1 +
                      (pcCol1 + iChannels)[i] * w2 +
                      pcCol2[i] * w3 +
                      (pcCol2 + iChannels)[i] * w4) * 0.000030517578125; 

      pcPixel += iChannels;

      xP += xP2;
    }
    yP += yP2;
  }
}

//===================================================================
//= Function name : void FastestResize
//= Description   : 
//= Return type   : template<class ElemType> 
//===================================================================

template <class ElemType>
void FastestResize(int in_iScrWdt,
                   int in_iSrcHgt,
                   ElemType** pSrcLines,
                   int in_iDstWdt,
                   int in_iDstHgt,
                   ElemType** pDstLines,
                   int iChannels)
{
  int x, y, xP, yP;

  ElemType* pcRead, * pcPixel;

  double dXScale = 1, dYScale = 1;
  int iXScale, iYScale;

  if (in_iDstWdt < 1 || in_iDstHgt < 1)
    return;

  if ((in_iScrWdt == in_iDstWdt) && (in_iSrcHgt == in_iDstHgt))
    return;

  dXScale = in_iDstWdt / (double) in_iScrWdt;
  dYScale = in_iDstHgt / (double) in_iSrcHgt;

  if ((dXScale < 1) || (dYScale < 1))
  {
    iXScale = (in_iScrWdt << 16) / in_iDstWdt;
    iYScale = (in_iSrcHgt << 16) / in_iDstHgt;
    yP = 0;
    for (y = 0; y <= in_iDstHgt - 1; y++)
    {
      xP = 0;
      pcRead = (ElemType *) pSrcLines[yP >> 16];
      pcPixel = (ElemType *) pDstLines[y];
      for (x = 0; x < in_iDstWdt; x++)
      {
        for (int i = 0; i < iChannels; i++)
          pcPixel[i] = pcRead[(xP >> 16) * iChannels + i];

        pcPixel += iChannels;
        xP += iXScale;
      }
      yP += iYScale;
    }
  }
  else
  {
    iYScale = (dYScale + 1);
    iXScale = (dXScale + 1);
    std::vector <ElemType> pcLine;
    pcLine.resize(in_iDstWdt * iChannels);

    for (y = 0; y <= in_iSrcHgt - 1; y++)
    {
      yP = dYScale * y;
      pcRead = (ElemType *) pSrcLines[y];
      for (x = 0; x < in_iScrWdt; x++)
      {
        xP = dXScale * x;
        for (int xCount = 0; xCount <= iXScale - 1; xCount++)
        {
          int xD = xCount + xP;
          if (xD >= in_iDstWdt)
            break;

          for (int i = 0; i < iChannels; i++)
            pcLine[xD * iChannels + i] = pcRead[x * iChannels + i];
        }
      }
      for (int yCount = 0; yCount <= iYScale - 1; yCount++)
      {
        int yD = yCount + yP;
        if (yD >= in_iDstHgt)
          break;
        memcpy(pDstLines[yD], &pcLine[0], in_iDstWdt * iChannels);
      }
    }
  }
}

//---------------------------------------------------------------------------

//===================================================================
//= Function name : FilterResize::Apply
//= Description   : 
//= Return type   : void 
//===================================================================

void FilterResize::Apply(const Image* pBmpSource, Image* pBmpDest) const
{
  if (m_iDstHgt < 1 || m_iDstHgt < 1)
    return;

  if (pBmpSource->GetWidth() == m_iDstWdt &&
      pBmpSource->GetHeight() == m_iDstHgt)
  {
    *pBmpDest = *pBmpSource;
    return;
  }

  pBmpDest->Create(m_iDstWdt,
                   m_iDstHgt,
                   pBmpSource->GetFormat(),
                   pBmpSource->GetRepres(),
                   pBmpSource->GetOrient());

  if (!pBmpDest->Created())
    return;

  BYTE** pSrcLines = pBmpSource->GetLineArray();
  BYTE** pDstLines = pBmpDest->GetLineArray();
  int iChannels = gml::Image::Format2Channels(pBmpSource->GetFormat());

  switch (m_ResizeType)
  {
    case FASTEST:
      {
        switch (pBmpSource->GetRepres())
        {
          case Image::R_BYTE:
            FastestResize<BYTE>(pBmpSource->GetWidth(),
                                pBmpSource->GetHeight(),
                                pSrcLines,
                                pBmpDest->GetWidth(),
                                pBmpDest->GetHeight(),
                                pDstLines,
                                iChannels); 
            break;
          case Image::R_WORD:
            FastestResize<WORD>(pBmpSource->GetWidth(),
                                pBmpSource->GetHeight(),
                                (WORD * *) pSrcLines,
                                pBmpDest->GetWidth(),
                                pBmpDest->GetHeight(),
                                (WORD * *) pDstLines,
                                iChannels); 
            break;
          case Image::R_FLOAT:
            FastestResize<float>(pBmpSource->GetWidth(),
                                 pBmpSource->GetHeight(),
                                 (float * *) pSrcLines,
                                 pBmpDest->GetWidth(),
                                 pBmpDest->GetHeight(),
                                 (float * *) pDstLines,
                                 iChannels); 
            break;
          case Image::R_DOUBLE:
            FastestResize<double>(pBmpSource->GetWidth(),
                                  pBmpSource->GetHeight(),
                                  (double * *) pSrcLines,
                                  pBmpDest->GetWidth(),
                                  pBmpDest->GetHeight(),
                                  (double * *) pDstLines,
                                  iChannels); 
            break;
          default:
            ASSERT(0 == "Not supported");
            break;
        };
      }
      break;
    case BILINEAR:
      switch (pBmpSource->GetRepres())
      {
        case Image::R_BYTE:
          BilinearResize<BYTE>(pBmpSource->GetWidth(),
                               pBmpSource->GetHeight(),
                               pSrcLines,
                               pBmpDest->GetWidth(),
                               pBmpDest->GetHeight(),
                               pDstLines,
                               iChannels); 
          break;
        case Image::R_WORD:
          BilinearResize<WORD>(pBmpSource->GetWidth(),
                               pBmpSource->GetHeight(),
                               (WORD * *) pSrcLines,
                               pBmpDest->GetWidth(),
                               pBmpDest->GetHeight(),
                               (WORD * *) pDstLines,
                               iChannels); 
          break;
        case Image::R_FLOAT:
          BilinearResize<float>(pBmpSource->GetWidth(),
                                pBmpSource->GetHeight(),
                                (float * *) pSrcLines,
                                pBmpDest->GetWidth(),
                                pBmpDest->GetHeight(),
                                (float * *) pDstLines,
                                iChannels); 
          break;
        case Image::R_DOUBLE:
          BilinearResize<double>(pBmpSource->GetWidth(),
                                 pBmpSource->GetHeight(),
                                 (double * *) pSrcLines,
                                 pBmpDest->GetWidth(),
                                 pBmpDest->GetHeight(),
                                 (double * *) pDstLines,
                                 iChannels); 
          break;
        default:
          ASSERT(0 == "Not supported");
          break;
      };
      break;
  }
}




