//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlfilterrepres.cpp,v 1.4 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#include "../../../base/gmlcommon.h"
#include "../../../math/gmlmath.h"
#include "../gmlimage.h"
#include "gmlFilterRepres.h"
#include <limits>
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

#undef min
#undef max

using gml::FilterRepres;
using gml::Image;

FilterRepres::FilterRepres(Image::REPRES in_Repres,
                           double in_dCoeff,
                           bool in_bClipTypeBounds) : ImageFilter()
{
  SetParam(in_Repres, in_dCoeff, in_bClipTypeBounds);
}

FilterRepres::~FilterRepres()
{
}

//===================================================================
//= Function name : SetParam
//= Description   : 
//= Return type   : void
//===================================================================

void FilterRepres::SetParam(Image::REPRES in_Repres,
                            double in_dCoeff,
                            bool in_bClipTypeBounds)
{
  m_dCoeff = in_dCoeff;
  m_bClipTypeBounds = in_bClipTypeBounds;
  m_DestRepres = in_Repres;
}

//===================================================================
//= Function name : ChangeRep
//= Description   : 
//= Return type   : void
//===================================================================

template <class T, class D>
void ChangeRep(int width,
               int height,
               T** pSrcLines,
               D** pDstLines,
               int in_iChannels,
               double in_dMult,
               bool in_bClipTypeBounds)
{
  if (in_bClipTypeBounds && !std::numeric_limits<D>::has_infinity)
  {
    for (int y = 0; y < height; ++y)
    {
      // For each line
      T* pSrcPixel = reinterpret_cast<T*>(pSrcLines[y]);
      D* pDstPixel = reinterpret_cast<D*>(pDstLines[y]);

      for (int x = 0; x < width*in_iChannels; ++x)
      {
        // For each pixel

        *pDstPixel = (D) gml::Clipped((in_dMult * *pSrcPixel),
                                      std::numeric_limits<D>::min(),
                                      std::numeric_limits<D>::max());
        pSrcPixel++;
        pDstPixel++;
      }
    }
  }
  else
  {
    for (int y = 0; y < height; ++y)
    {
      // For each line
      T* pSrcPixel = reinterpret_cast<T*>(pSrcLines[y]);
      D* pDstPixel = reinterpret_cast<D*>(pDstLines[y]);

      for (int x = 0; x < width*in_iChannels; ++x)
      {
        // For each pixel
        *pDstPixel = (D) (in_dMult * *pSrcPixel);
        pSrcPixel++;
        pDstPixel++;
      }
    }
  }
}

//===================================================================
//= Function name : UglySort
//= Description   : 
//= Return type   : void
//===================================================================

template <class T>
void UglySort(T** in_pSrcLines,
              Image* pBmpDest,
              Image::REPRES in_Repres,
              double in_dCoeff,
              bool in_bClipTypeBounds)
{
  int iChannels = Image::Format2Channels(pBmpDest->GetFormat());
  BYTE** pDstLines = pBmpDest->GetLineArray();

  switch (in_Repres)
  {
    case Image::R_BYTE:
      ChangeRep<T, BYTE>(pBmpDest->GetWidth(),
                         pBmpDest->GetHeight(),
                         in_pSrcLines,
                         (BYTE * *) pDstLines,
                         iChannels,
                         in_dCoeff,
                         in_bClipTypeBounds); 
      break;
    case Image::R_WORD:
      ChangeRep<T, WORD>(pBmpDest->GetWidth(),
                         pBmpDest->GetHeight(),
                         in_pSrcLines,
                         (WORD * *) pDstLines,
                         iChannels,
                         in_dCoeff,
                         in_bClipTypeBounds); 
      break;
    case Image::R_FLOAT:
      ChangeRep<T, float>(pBmpDest->GetWidth(),
                          pBmpDest->GetHeight(),
                          in_pSrcLines,
                          (float * *) pDstLines,
                          iChannels,
                          in_dCoeff,
                          in_bClipTypeBounds); 
      break;
    case Image::R_DOUBLE:
      ChangeRep<T, double>(pBmpDest->GetWidth(),
                           pBmpDest->GetHeight(),
                           in_pSrcLines,
                           (double * *) pDstLines,
                           iChannels,
                           in_dCoeff,
                           in_bClipTypeBounds);
      break;
    default:
      ASSERT(0 == "Not supported");
      break;
  };
}

//===================================================================
//= Function name : Apply
//= Description   : 
//= Return type   : void
//===================================================================

void FilterRepres::Apply(const Image* pBmpSource, Image* pBmpDest) const
{
  pBmpDest->Create(pBmpSource->GetWidth(),
                   pBmpSource->GetHeight(),
                   pBmpSource->GetFormat(),
                   m_DestRepres,
                   pBmpSource->GetOrient());

  BYTE** pSrcLines = pBmpSource->GetLineArray();

  switch (pBmpSource->GetRepres())
  {
    case Image::R_BYTE:
      UglySort<BYTE>((BYTE * *) pSrcLines,
                     pBmpDest,
                     m_DestRepres,
                     m_dCoeff,
                     m_bClipTypeBounds); 
      break;
    case Image::R_WORD:
      UglySort<WORD>((WORD * *) pSrcLines,
                     pBmpDest,
                     m_DestRepres,
                     m_dCoeff,
                     m_bClipTypeBounds); 
      break;
    case Image::R_FLOAT:
      UglySort<float>((float * *) pSrcLines,
                      pBmpDest,
                      m_DestRepres,
                      m_dCoeff,
                      m_bClipTypeBounds); 
      break;
    case Image::R_DOUBLE:
      UglySort<double>((double * *) pSrcLines,
                       pBmpDest,
                       m_DestRepres,
                       m_dCoeff,
                       m_bClipTypeBounds); 
      break;
    default:
      ASSERT(0 == "Not supported");
      break;
  };
}


