//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlfiltergrayscale.cpp,v 1.17 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#include "../../../base/gmlcommon.h"
#include "../gmlimage.h"
#include "gmlfiltergrayscale.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

using gml::FilterGrayscale;
using gml::Image;


//=============================================================================================
///////////////////////////////////////////////////////////////////////////////////////////////
//   Color to Gray
///////////////////////////////////////////////////////////////////////////////////////////////
//=============================================================================================

FilterGrayscale::FilterGrayscale() : ImageFilter()
{
  m_method = M_EYE;
}

FilterGrayscale::~FilterGrayscale()
{
}

namespace
{
  template <class T>
  class Do
  {
    public:
      void operator()(int width,
                      int height,
                      BYTE** pSrcLines,
                      BYTE** pDstLines,
                      FilterGrayscale::METHOD m) const
      {
        for (int y = 0; y < height; ++y)
        {
          // For each line
          T* pSrcPixel = reinterpret_cast<T*>(pSrcLines[y]);
          T* pDstPixel = reinterpret_cast<T*>(pDstLines[y]);

          for (int x = 0; x < width ; ++x)
          {
            switch (m)
            {
              case FilterGrayscale::M_AVERAGE:
                {
                  *pDstPixel = (T)
                               ((pSrcPixel[0] +
                                 (float) pSrcPixel[1] +
                                 (float) pSrcPixel[2]) /
                                3);
                  break;
                }

              case FilterGrayscale::M_R:
                {
                  *pDstPixel = pSrcPixel[0]; // FIXME different result for RGB and BGR images
                  break;
                }

              case FilterGrayscale::M_G:
                {
                  *pDstPixel = pSrcPixel[1]; // FIXME different result for RGB and BGR images
                  break;
                }

              case FilterGrayscale::M_B:
                {
                  *pDstPixel = pSrcPixel[2]; // FIXME different result for RGB and BGR images
                  break;
                }
            }
            pSrcPixel += 3; // SOURCE IS RGB
            pDstPixel += 1; // DEST IS LUM
          }
        };
      };
  };
}


//===================================================================
//= Function name : FastRGBToGray
//= Description   : 
//= Return type   : void 
//===================================================================

template <class ElemType>
void FastColorToGray(int in_iWdt,
                     int in_iHgt,
                     ElemType** pSrcLines,
                     ElemType** pDstLines,
                     gml::Image::FORMAT in_Format)
{
  if (in_Format == Image::F_BGR)
  {
    for (int y = 0; y < in_iHgt; ++y)
    {
      ElemType* pSrcPixel = reinterpret_cast<ElemType*>(pSrcLines[y]);
      ElemType* pDstPixel = reinterpret_cast<ElemType*>(pDstLines[y]);

      for (int x = 0; x < in_iWdt ; ++x)
      {
        // For the coefficients used, see http://www.inforamp.net/~poynton/
        // We could approximate this for more speed by using
        // Y = (54 * R + 183 * G + 19 * B)/256 like libpng does.
        *pDstPixel = (ElemType) pSrcPixel[2] * 0.299 +
                     pSrcPixel[1] * 0.587 +
                     pSrcPixel[0] * 0.114;
        //(ElemType) pSrcPixel[2] * 0.212671 + pSrcPixel[1] * 0.715160 + pSrcPixel[0] * 0.072169;

        pSrcPixel += 3; // SOURCE IS RGB
        pDstPixel += 1; // DEST IS LUM
      };
    };
  }
  else
  {
    for (int y = 0; y < in_iHgt; ++y)
    {
      ElemType* pSrcPixel = reinterpret_cast<ElemType*>(pSrcLines[y]);
      ElemType* pDstPixel = reinterpret_cast<ElemType*>(pDstLines[y]);

      for (int x = 0; x < in_iWdt ; ++x)
      {
        // For the coefficients used, see http://www.inforamp.net/~poynton/
        // We could approximate this for more speed by using
        // Y = (54 * R + 183 * G + 19 * B)/256 like libpng does.
        *pDstPixel = (ElemType) pSrcPixel[0] * 0.212671 +
                     pSrcPixel[1] * 0.715160 +
                     pSrcPixel[2] * 0.072169;

        pSrcPixel += 3; // SOURCE IS RGB
        pDstPixel += 1; // DEST IS LUM
      };
    };
  }
}


//===================================================================
//= Function name : FilterGrayscale::Apply
//= Description   : 
//= Return type   : void 
//===================================================================

void FilterGrayscale::Apply(const Image* pBmpSource, Image* pBmpDest) const
{
  ASSERT(pBmpSource->GetFormat() == Image::F_RGB ||
         pBmpSource->GetFormat() == Image::F_BGR);

  pBmpDest->Create(pBmpSource->GetWidth(),
                   pBmpSource->GetHeight(),
                   Image::F_L,
                   pBmpSource->GetRepres(),
                   pBmpSource->GetOrient());
  BYTE** pSrcLines = pBmpSource->GetLineArray();
  BYTE** pDstLines = pBmpDest->GetLineArray();

  if (m_method == M_EYE)
  {
    switch (pBmpSource->GetRepres())
    {
      case Image::R_BYTE:
        FastColorToGray<BYTE>(pBmpDest->GetWidth(),
                              pBmpDest->GetHeight(),
                              pSrcLines,
                              pDstLines,
                              pBmpSource->GetFormat()); 
        break;
      case Image::R_WORD:
        FastColorToGray<WORD>(pBmpDest->GetWidth(),
                              pBmpDest->GetHeight(),
                              (WORD * *) pSrcLines,
                              (WORD * *) pDstLines,
                              pBmpSource->GetFormat()); 
        break;
      case Image::R_FLOAT:
        FastColorToGray<float>(pBmpDest->GetWidth(),
                               pBmpDest->GetHeight(),
                               (float * *) pSrcLines,
                               (float * *) pDstLines,
                               pBmpSource->GetFormat()); 
        break;
      case Image::R_DOUBLE:
        FastColorToGray<double>(pBmpDest->GetWidth(),
                                pBmpDest->GetHeight(),
                                (double * *) pSrcLines,
                                (double * *) pDstLines,
                                pBmpSource->GetFormat()); 
        break;
    };
  }
  else
  {
    switch (pBmpSource->GetRepres())
    {
      case Image::R_BYTE:
        Do<BYTE>()(pBmpDest->GetWidth(),
                   pBmpDest->GetHeight(),
                   pSrcLines,
                   pDstLines,
                   m_method); 
        break;
      case Image::R_WORD:
        Do<WORD>()(pBmpDest->GetWidth(),
                   pBmpDest->GetHeight(),
                   pSrcLines,
                   pDstLines,
                   m_method); 
        break;
      case Image::R_FLOAT:
        Do<float>()(pBmpDest->GetWidth(),
                    pBmpDest->GetHeight(),
                    pSrcLines,
                    pDstLines,
                    m_method); 
        break;
      case Image::R_DOUBLE:
        Do<double>()(pBmpDest->GetWidth(),
                     pBmpDest->GetHeight(),
                     pSrcLines,
                     pDstLines,
                     m_method); 
        break;
    };
  };
}










