//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: IntelImage.h,v 1.16 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

//////////////////////////////////////////////////////////////////////
//
// IntelImage class - encapsulates IplImage,
// can draw itself, save andf load from file.
// Also includes siplified clone and create functions
//
//////////////////////////////////////////////////////////////////////

#if !defined(AFX_DINTELIMAGE_H__16CDF827_AE48_11D4_90F0_00C04F791DEA__INCLUDED_)
#define AFX_DINTELIMAGE_H__16CDF827_AE48_11D4_90F0_00C04F791DEA__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#ifdef GML_USE_MFC
#include <afxwin.h>
#endif

#include <ipl.h>

#include "..\..\Math\gmlBBox2.h"
#include "..\bitmap\gmlsimpleimage.h"

#ifdef GML_USE_OWL
#include <windows.hpp>
#include <ExtCtrls.hpp>
#endif


/** @addtogroup Images
 *  @{
 */

/// Intel libraries IplImage wrap class
class DIntelImage
{
  protected:

    /// Pointer to stored IplImage
    IplImage* m_pImage;

    /// @name  Forbidden operations
    /// @{

    /// Copy constructor (not implemented)
    DIntelImage(DIntelImage&);

    /// Assignment operator (not implemented)
    DIntelImage operator =(DIntelImage&);
    /// @}


  public:

    /// Free image memory (called automatically)
    void DeleteImages();

    /// Check if image matches the specified size and channels number
    inline bool CheckImage(int width, int height, int in_iChannels)
    {
      if (m_pImage &&
          (m_pImage->width == width) &&
          (m_pImage->height == height) &&
          (m_pImage->depth * m_pImage->nChannels == in_iChannels))
        return true;
      else
        return false;
    };


    /// @name  Creation
    /// @{

    /// Default constructor
    DIntelImage();

    /// Create an empty image with specified size, bit depth and channels number
    void CreateImage(int width,
                     int height,
                     int depth = IPL_DEPTH_8U,
                     int nChannels = 3);

    /// Attach IplImage. The image is replaced by in_pImage, memory management will be done by DIntelImage from now on.
    void AttachImage(IplImage* in_pImage);

    /// Create a full copy of the in_pImage
    void CloneImage(IplImage* in_pImage);

    /// Create a copy of the given DIntelImage with possibility of color <-> grayscale transfer

    /** @param in_Image image to copy;
        @param desired_color contols the color properties of the resulting image (-1 - do not change color/grayscale properties, 
        0 - the image becomes grayscale, 1 - image becomes color).
    */

    void CopyOf(DIntelImage& in_Image, int desired_color = -1);
    /// @}


    /// @name Relation to SimpleImage
    /// @{ 
    gml::SimpleImage* CopyToSimpleImage();
    /// @}

    /// @name  Destruction
    /// @{

    /// Detach IplImage. The detached IplImage should be destroyed by the programmer.
    IplImage* DetachImage();

    /// Destructor
    virtual ~DIntelImage();
    /// @}

    /// @name  Filework
    /// @{

    /// Load image from file

    /** Load image from a specified filename. Returns true in case of success, false on error. 
        Depends on the highgui.dll and (c)vlgrfmts.dll installed on your machine (both libraries
        are provided with Intel OpenCV), but usually jpeg, bmp, tiff, pxm are supported.
    */
    bool LoadImage(const char* in_sFileName);

    /// Save image to file

    /** Save image to a specified filename. The image format is taken from the file extension (jpg, bmp, tif, pxm).
        Depends on the highgui.dll and (c)vlgrfmts.dll installed 
        on your machine (both libraries are provided with Intel OpenCV), but usually jpeg, bmp, tiff, 
        pxm are supported.
    */
    void SaveImage(const char* in_sFileName);
    /// @}


#ifdef GML_USE_MFC
    /// @name  Image drawing (actually, painting) in MFC 
    /// @{

    /// Paint image in MFC device context with specified offset and scale

    /** Function is available only if GML_USE_MFC is defined at the time of compilation. The image is scaled to the given ratio and
        painted in the CDC with the specified offset
    */
    void DrawImage(CDC* in_pDC, CPoint in_vOrigin, float in_dRatio);

    /// Paint image in MFC device context with specified offset and destination image height

    /** Function is available only if GML_USE_MFC is defined at the time of compilation. The image is scaled to fit the given height 
        (aspect ratio is preserved) and  painted in the CDC with the specified offset
    */
    void DrawImage(CDC* in_pDC, CPoint in_vOrigin, int in_iHeight = -1);

    /// Paint part of the image in MFC device context in specified window

    /** Function is available only if GML_USE_MFC is defined at the time of compilation. The image is scaled to fit the given window 
        (aspect ratio is preserved)
    */

    void DrawImage(CDC* in_pDC, gml::BBox2i in_SrcBox, gml::BBox2i in_TrgBox);
    /// @}
#endif

#ifdef GML_USE_OWL
    /// @name  Image drawing (actually, painting) in Borland Object Window Library 
    /// @{

    /// Paint image in Borland OWL TBitmap with specified offset and scale

    /** Function is available only if GML_USE_OWL is defined at the time of compilation. 
        The image is scaled to the given ratio and
        painted in the TBitmap with the specified offset
    */
    void DrawImage(Graphics::TBitmap* in_pTarget,
                   TPoint in_vOrigin,
                   float in_dRatio = 1);
    /// @}
#endif


    /// Get pointer to stored image
    inline IplImage* GetImage()
    {
      return m_pImage;
    };

    /// Get width of the image
    inline int width()
    {
      if (m_pImage != 0)
        return m_pImage->width;
      else
        return -1;
    };

    /// Get height of the image
    inline int height()
    {
      if (m_pImage != 0)
        return m_pImage->height;
      else
        return -1;
    };
    /// Get image aligned width
    inline int step()
    {
      if (m_pImage != 0)
        return m_pImage->widthStep;
      else
        return -1;
    };

    /// Get pointer to line
    inline unsigned char* GetLineFast(int in_iLine)
    {
      return (unsigned char *) m_pImage->imageData +
             m_pImage->widthStep * in_iLine;
    };

    /// Get pointer to line
    inline unsigned char* GetLineSafe(int in_iLine)
    {
      if ((m_pImage != 0) && (in_iLine >= 0) && (in_iLine < m_pImage->height))
        return (unsigned char *) m_pImage->imageData +
               m_pImage->widthStep * in_iLine;
      else
        return NULL;
    };
};


/** @} */


#ifdef GML_USE_MFC
/// Paint IplImage in MFC device context with specified offset and scale

/** Function is available only if GML_USE_MFC is defined at the time of compilation. The image is scaled to the given ratio and
    painted in the CDC with the specified offset
*/
void PaintImage(IplImage* in_pImage,
                CDC* in_pDC,
                CPoint in_vOrigin,
                float in_dRatio);

/// Paint part of the image in MFC device context in specified window

/** Function is available only if GML_USE_MFC is defined at the time of compilation. The image is scaled to fit the given window 
    (aspect ratio is preserved)
*/

void PaintImage(IplImage* in_pImage,
                CDC* in_pDC,
                gml::BBox2i in_SrcBox,
                gml::BBox2i in_TrgBox);

/// @}
#endif

/** Create a new image scaled to given ration and return pointer to it
*/
IplImage* ScaleImage(IplImage* in_pImage, double in_dScale);

#endif // !defined(AFX_DINTELIMAGE_H__16CDF827_AE48_11D4_90F0_00C04F791DEA__INCLUDED_)
