//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: IntelImage.cpp,v 1.27 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

//////////////////////////////////////////////////////////////////////
//
// IntelImage class - encapsulates IplImage, 
// can draw itself, save andf load from file.
// Also includes siplified clone and create functions
//                  
//////////////////////////////////////////////////////////////////////

#ifdef GML_USE_MFC
#include <afxwin.h>
#endif

#include "IntelImage.h"
#include <highgui.h>
#include <assert.h>
#include <cv.h>

#ifdef GML_USE_MFC
void atsGetImageInfo(IplImage* img,
                     void** pData,
                     int* pStep,
                     CSize* pSz,
                     int* pDepth,
                     int* pChannels,
                     int* pBtPix);
void atsDisplayImage(IplImage* img,
                     CDC* in_pDC,
                     CPoint dst_org,
                     CSize dst_size);
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

DIntelImage::DIntelImage()
{
  m_pImage = NULL;
  //m_pDib = NULL;
} 

DIntelImage::~DIntelImage()
{
  DeleteImages();
} 


//===================================================================
//= Function name : DIntelImage::DeleteImages
//= Description   : 
//= Return type   : void 
//===================================================================

void DIntelImage::DeleteImages()
{
  if (m_pImage)
    cvReleaseImage(&m_pImage);
}


//===================================================================
//= Function name : DIntelImage::LoadImage
//= Description   :
//= Return type   : void
//===================================================================

bool DIntelImage::LoadImage(LPCTSTR in_sFileName)
{
  DeleteImages();

  m_pImage = cvvLoadImage(in_sFileName);
  if (!m_pImage)
    return false;
  else
    return true;
}


//===================================================================
//= Function name : DIntelImage::CreateImage
//= Description   :
//= Return type   : void 
//===================================================================

void DIntelImage::CreateImage(int width, int height, int depth, int nChannels)
{
  DeleteImages();

  CvSize size;

  size.width = width;
  size.height = height;

  m_pImage = cvCreateImage(size, depth, nChannels);
}



//===================================================================
//= Function name : DIntelImage::SaveImage
//= Description   : 
//= Return type   : void 
//===================================================================

void DIntelImage::SaveImage(LPCTSTR in_sFileName)
{
  if (!m_pImage)
    return;

  if (m_pImage->nChannels == 3)
    cvvSaveImage(in_sFileName, m_pImage);
  else
  {
    DIntelImage TempImage;
    TempImage.CreateImage(width(), height());

    iplGrayToColor(m_pImage, TempImage.GetImage(), 1, 1, 1);
    TempImage.SaveImage(in_sFileName);
  }
}



//===================================================================
//= Function name : DIntelImage::DrawImage
//= Description   : 
//= Return type   : void 
//===================================================================

gml::SimpleImage* DIntelImage::CopyToSimpleImage()
{
  gml::SimpleImage* pSimpleImage = new gml::SimpleImage;
  pSimpleImage->Create(width(),
                       height(),
                       gml::Image::F_RGB,
                       gml::Image::R_BYTE);

  unsigned char * *ppucLines = pSimpleImage->GetLineArray();

  for (int i = 0; i < height(); i++)
  {
    unsigned char * pucLine = GetLineSafe(height() - 1 - i);

    for (int j = 0; j < width() * 3 ; j += 3)
    {
      ppucLines[i][j] = pucLine[j + 2];
      ppucLines[i][j + 1] = pucLine[j + 1];
      ppucLines[i][j + 2] = pucLine[j];
    }
  }

  return pSimpleImage;

  //  return NULL;
}


//===================================================================
//= Function name : DIntelImage::DrawImage
//= Description   : 
//= Return type   : void 
//===================================================================

#ifdef GML_USE_MFC
void DIntelImage::DrawImage(CDC* in_pDC, CPoint in_vOrigin, float in_dRatio)
{
  PaintImage(m_pImage, in_pDC, in_vOrigin, in_dRatio);
}

void DIntelImage::DrawImage(CDC* in_pDC, CPoint in_vOrigin, int in_iHeight)
{
  double dRatio = 1;

  if (!m_pImage)
    return;

  if (in_iHeight > 1)
    dRatio = in_iHeight / (double) m_pImage->height;

  PaintImage(m_pImage, in_pDC, in_vOrigin, (float) (dRatio));
}

void DIntelImage::DrawImage(CDC* in_pDC,
                            gml::BBox2i in_SrcBox,
                            gml::BBox2i in_TrgBox)
{
  PaintImage(GetImage(), in_pDC, in_SrcBox, in_TrgBox);
}

void PaintImage(IplImage* in_pImage,
                CDC* in_pDC,
                gml::BBox2i in_SrcBox,
                gml::BBox2i in_TrgBox)
{
  double dRatio = in_TrgBox.Width() / (double) in_SrcBox.Width();

  if (!in_pImage)
    return;

  if ((dRatio < 0.001) || (dRatio > 1000))
    return;

  // Crop and paint image
  CPoint vOrigin = in_TrgBox.vmin;

  IplROI roi = cvRectToROI(in_SrcBox);

  DIntelImage TempImage;
  TempImage.CreateImage(in_SrcBox.Width(),
                        in_SrcBox.Height(),
                        in_pImage->depth,
                        in_pImage->nChannels);

  in_pImage->roi = &roi;
  iplCopy(in_pImage, TempImage.GetImage());
  in_pImage->roi = NULL;


  TempImage.DrawImage(in_pDC, vOrigin, (float) (dRatio));
}

#endif //GML_USE_MFC

#ifdef GML_USE_OWL

//===================================================================
//= Function name : DIntelImage::DrawImage
//= Description   : Variant for BorlandC
//= Return type   : void 
//===================================================================

void DIntelImage::DrawImage(Graphics::TBitmap* in_pTarget,
                            TPoint in_vOrigin,
                            float in_fRatio)
{
  if (!m_pImage)
    return;

  int iWdt = m_pImage-> width, iHgt = m_pImage->height;

  DIntelImage tempImage1, tempImage2;

  IplImage* pImage = m_pImage;

  if ((m_pImage->depth != IPL_DEPTH_8U && m_pImage->depth != IPL_DEPTH_8S))
  {
    CvSize Size;

    Size.width = m_pImage->width;
    Size.height = m_pImage->height;

    tempImage1.CreateImage(m_pImage->width, m_pImage->height);
    pImage = tempImage1.GetImage();
    iplConvert(m_pImage, pImage);
  };

  if (in_fRatio < 1)
  {
    CvSize Size1;

    Size1.width = pImage->width * in_fRatio;
    Size1.height = pImage->height * in_fRatio;

    tempImage2.CreateImage(Size1.width,
                           Size1.height,
                           IPL_DEPTH_8U,
                           pImage->nChannels); //create image header

    iplDecimate(pImage,
                tempImage2.GetImage(),
                Size1.width,
                pImage->width,
                Size1.height,
                pImage->height,
                IPL_INTER_NN);

    pImage = tempImage2.GetImage();
  }
  else
  {
  }

  in_pTarget->Width = pImage->width;
  in_pTarget->Height = pImage->height;
  in_pTarget->PixelFormat = pf24bit;

  char* src_data = pImage->imageData;
  int src_step = pImage->widthStep;

  assert(pImage->nChannels == 1 || pImage->nChannels == 3);

  /* convert source image to RGB24 */
  for (int y = 0; y < pImage->height; y++, src_data += src_step)
  {
    memcpy(in_pTarget->ScanLine[y], src_data, src_step);
  }
}

#endif // GML_USE_OWL

//===================================================================
//= Function name : DIntelImage::CloneImage
//= Description   :
//= Return type   : void
//===================================================================

void DIntelImage::CloneImage(IplImage* in_pImage)
{
  IplImage* pTempImage = cvCloneImage(in_pImage);

  if (m_pImage)
    DeleteImages();

  m_pImage = pTempImage;

  int iError;
  if (iError = iplGetErrStatus())
    DeleteImages();
}

#ifdef GML_USE_MFC


//===================================================================
//= Function name : PaintImage
//= Description   : 
//= Return type   : void 
//===================================================================

void PaintImage(IplImage* in_pImage,
                CDC* in_pDC,
                CPoint in_vOrigin,
                float in_dRatio)
{
  IplImage* pImage;

  if (!in_pImage || !in_pDC)
    return;

  CSize size;

  if ((in_pImage->depth != IPL_DEPTH_8U && in_pImage->depth != IPL_DEPTH_8S))
  {
    CvSize Size;

    Size.width = in_pImage->width;
    Size.height = in_pImage->height;

    pImage = cvCreateImage(Size, IPL_DEPTH_8U, in_pImage->nChannels);
    iplConvert(in_pImage, pImage);
  }
  else
    pImage = in_pImage;


  size.cx = (long) (in_pImage->width * in_dRatio);
  size.cy = (long) (in_pImage->height * in_dRatio);

  atsDisplayImage(pImage, in_pDC, in_vOrigin, size);

  if (pImage != in_pImage)
    cvReleaseImage(&pImage);
}

//===================================================================
//= Function name : atsGetImageInfo
//= Description   :
//= Return type   : void
//===================================================================

void atsGetImageInfo(IplImage* img,
                     void** pData,
                     int* pStep,
                     CSize* pSz,
                     int* pDepth,
                     int* pChannels,
                     int* pBtPix)
{
  if (!img)
  {
    assert(0);
  }
  else
  {
    int channels = img->nChannels;
    int depth = img->depth;
    int step = img->widthStep;
    int bt_pix;

    if (img->origin != IPL_ORIGIN_TL || img->dataOrder != IPL_DATA_ORDER_PIXEL)
    {
      assert(0);
      return;
    }

    bt_pix = ((depth & 255) >> 3) * channels;

    if (pDepth)
      *pDepth = depth;
    if (pChannels)
      *pChannels = channels;
    if (pStep)
      *pStep = step;
    if (pBtPix)
      *pBtPix = bt_pix;

    if (pSz)
    {
      pSz->cx = img->roi ? img->roi->width : img->width;
      pSz->cy = img->roi ? img->roi->height : img->height;
    }

    if (pData)
    {
      *pData = img->imageData +
               (!img->roi ?
                0 :
                img->roi->yOffset * step +
                img->roi->xOffset * bt_pix);
    }
  }
}


//===================================================================
//= Function name : atsDisplayImage
//= Description   : displays (8u/8s)C1/C3/AC4 image
//= Return type   : void  
//===================================================================

void atsDisplayImage(IplImage* img,
                     CDC* in_pDC,
                     CPoint dst_org,
                     CSize dst_size)
{
  uchar storage[sizeof(BITMAPINFOHEADER) + 256 * 4];
  BITMAPINFOHEADER* bmih = (BITMAPINFOHEADER*) storage;

  char* src_data = 0;
  char* dst_data = 0;
  CSize src_size;
  int src_step = 0;
  int channels = 0, depth = 0, bt_pix = 0;
  HBITMAP hbmp = 0;
  HDC hdc = 0;

  atsGetImageInfo(img,
                  (void * *) &src_data,
                  &src_step,
                  &src_size,
                  &depth,
                  &channels,
                  &bt_pix);

  assert(depth == IPL_DEPTH_8U || depth == IPL_DEPTH_8S);
  assert(channels == 1 || channels == 3 || channels == 4);

  bmih->biSize = sizeof(*bmih);
  bmih->biWidth = src_size.cx;
  bmih->biHeight = -src_size.cy;
  bmih->biPlanes = 1;
  bmih->biBitCount = (unsigned short) (bt_pix * 8);
  bmih->biCompression = BI_RGB;
  bmih->biSizeImage = 0;
  bmih->biXPelsPerMeter = 0;
  bmih->biYPelsPerMeter = 0;
  bmih->biClrUsed = 0;
  bmih->biClrImportant = 0;

  if (channels == 1)
  {
    int i, delta = (depth == IPL_DEPTH_8S) * 128;
    RGBQUAD* palette = (RGBQUAD*) (storage + sizeof(BITMAPINFOHEADER));

    for (i = 0; i < 256; i++)
      palette[i].rgbBlue = palette[i].rgbRed = palette[i].rgbGreen = (uchar)
                           (i + delta);
  }

  hdc = CreateCompatibleDC(0);
  hbmp = CreateDIBSection(hdc,
                          (BITMAPINFO *) bmih,
                          DIB_RGB_COLORS,
                          (void * *) &dst_data,
                          0,
                          0);

  if (hbmp && in_pDC)
  {
    HGDIOBJ hold = SelectObject(hdc, hbmp);
    int dst_step = (src_size.cx* bt_pix + 3) & -4;
    int y;

    // convert source image to RGB24 
    for (y = 0;
         y < src_size.cy;
         y++, src_data += src_step, dst_data += dst_step)
      memcpy(dst_data, src_data, dst_step);

    SetStretchBltMode(in_pDC->m_hDC, COLORONCOLOR);
    StretchBlt(in_pDC->m_hDC,
               dst_org.x,
               dst_org.y,
               dst_size.cx,
               dst_size.cy,
               hdc,
               0,
               0,
               src_size.cx,
               src_size.cy,
               SRCCOPY);
    SelectObject(hdc, hold);
  }

  if (hbmp)
    DeleteObject(hbmp);
  if (hdc)
    DeleteDC(hdc);
}
#endif


//===================================================================
//= Function name : DIntelImage::CopyOf
//= Description   : 
//= Return type   : void 
//===================================================================

void DIntelImage::CopyOf(DIntelImage& image, int desired_color)
{
  IplImage* img = image.GetImage();
  if (img)
  {
    int color = desired_color;
    if (color < 0)
    {
      color = img->nChannels;
      CreateImage(img->width, img->height, IPL_DEPTH_8U, img->nChannels);
    }
    else if (color == 0)
    {
      color = 1;
      CreateImage(img->width, img->height, IPL_DEPTH_8U, 1);
    }
    else
    {
      color = 3;
      CreateImage(img->width, img->height, IPL_DEPTH_8U, 3);
    }

    if (m_pImage->nChannels == img->nChannels)
      iplCopy(img, m_pImage);
    else
    {
      if (color > 1)
        iplGrayToColor(img, m_pImage, 1, 1, 1);
      else
        iplColorToGray(img, m_pImage);
    }
  }
}


//===================================================================
//= Function name : DIntelImage::AttachImage
//= Description   : 
//= Return type   : void 
//===================================================================

void DIntelImage::AttachImage(IplImage* in_pImage)
{
  DeleteImages();
  m_pImage = in_pImage;
}


//===================================================================
//= Function name : *DIntelImage::DetachImage
//= Description   : 
//= Return type   : IplImage 
//===================================================================

IplImage* DIntelImage::DetachImage()
{
  IplImage* pToReturn = m_pImage;
  m_pImage = NULL;
  return pToReturn;
}



//===================================================================
//= Function name : ScaleImage
//= Description   : 
//= Return type   : void 
//===================================================================

IplImage* ScaleImage(IplImage* in_pImage, double in_dScale)
{
  int iXDst = in_pImage->width* in_dScale,
                                iYDst = in_pImage->height* in_dScale;

  IplImage* out_pImage = cvCreateImage(cvSize(iXDst, iYDst),
                                       in_pImage->depth,
                                       in_pImage->nChannels);


  if (in_dScale < 1)
    iplDecimate(in_pImage,
                out_pImage,
                iXDst,
                in_pImage->width,
                iYDst,
                in_pImage->height,
                IPL_INTER_CUBIC);
  else if (in_dScale > 1)
    iplZoom(in_pImage,
            out_pImage,
            iXDst,
            in_pImage->width,
            iYDst,
            in_pImage->height,
            IPL_INTER_CUBIC);
  else
    cvCopy(in_pImage, out_pImage);

  return out_pImage;
}

