//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlfrimageloader.cpp,v 1.8 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#include "../../base/gmlcommon.h"
#include "../bitmap/gmlimage.h"
#include "gmlfrimageloader.h"

#include <freeimage.h>

bool gml::FRImageLoader::EnumerateLoadableFormats(void (*f)
                                                  (IMAGE_FORMAT format))
{
  f(FORMAT_BMP);
  f(FORMAT_ICO);
  f(FORMAT_JPEG); 
  f(FORMAT_PNG);
  f(FORMAT_PGM);
  f(FORMAT_PGMRAW);
  f(FORMAT_PPM);
  f(FORMAT_TIFF); 
  f(FORMAT_TARGA); 
  return true;
}

bool gml::FRImageLoader::EnumerateSaveableFormats(void (*f)
                                                  (IMAGE_FORMAT format))
{
  f(FORMAT_BMP);
  f(FORMAT_ICO);
  f(FORMAT_JPEG); 
  f(FORMAT_PNG);
  f(FORMAT_PGM);
  f(FORMAT_PGMRAW);
  f(FORMAT_PPM);
  f(FORMAT_TIFF); 
  f(FORMAT_TARGA); 
  return true;
}

gml::ImageLoader::IMAGE_FORMAT gml::FRImageLoader::CheckBitMapFile(const std::string& path)
{
  //FREE_IMAGE_FORMAT f;
  FREE_IMAGE_FORMAT f = FreeImage_GetFileType(path.c_str(), 16);
  if (f == FIF_UNKNOWN)
    f = FreeImage_GetFIFFromFilename(path.c_str());
  //FreeImage
  return MapFreeImageFormatToOwnFormat(f);
}

bool gml::FRImageLoader::LoadBitMap(const std::string& path,
                                    Image& out_bitmap,
                                    IMAGE_FORMAT format /* = FORMAT_UNKNOWN */,
                                    int flags /* = 0 */)
{
  FREE_IMAGE_FORMAT file_format = FIF_UNKNOWN;
  if (format == FORMAT_UNKNOWN)
    format = CheckBitMapFile(path);

  file_format = MapOwnFormatToFreeImageFormat(format);
  FIBITMAP* fi_bitmap = FreeImage_Load(file_format, path.c_str());
  if (fi_bitmap == NULL)
    return false;

  int w = FreeImage_GetWidth(fi_bitmap);
  int h = FreeImage_GetHeight(fi_bitmap);
  int bpp = FreeImage_GetBPP(fi_bitmap);
  int size = w* h* bpp / 8;

  Image::FORMAT f;
  Image::REPRES s;
  switch (FreeImage_GetColorType(fi_bitmap))
  {
    case FIC_MINISBLACK:
      f = Image::F_L;
      s = (bpp == 8) ? Image::R_BYTE : Image::R_WORD;
      break;
    case FIC_RGB:
      f = Image::F_RGB;
      ASSERT(bpp == 24);
      s = Image::R_BYTE;
      break;
    case FIC_RGBALPHA:
      ASSERT(bpp == 32);
      f = Image::F_RGBA;
      s = Image::R_BYTE;
      break;
    case FIC_PALETTE:
      return false; // not yet supported
    default:
      ASSERT(FALSE);
  };

  out_bitmap.Create(w, h, f, s);
  out_bitmap.Lock(false, true);
  BYTE** line_array = out_bitmap.GetLineArray();
  int bytesperline = out_bitmap.GetBytesPerLine();
  for (int y = 0; y < h; ++y)
  {
    memcpy(line_array[y], FreeImage_GetScanLine(fi_bitmap, y), bytesperline);
  }

  out_bitmap.Unlock();

  if (out_bitmap.GetFormat() == gml::Image::F_RGB ||
      out_bitmap.GetFormat() == gml::Image::F_RGBA)
  {
    out_bitmap.Lock(true, true);

    for (int y = 0; y < h; ++y)
    {
      BYTE* p = line_array[y];
      for (int x = 0; x < w; ++x, p += out_bitmap.GetElemSize())
        std::swap(*p, *(p + 2));
    }

    out_bitmap.Unlock();
  }


  FreeImage_Unload(fi_bitmap);
  return true;
}

bool gml::FRImageLoader::SaveBitMap(const std::string& path,
                                    const Image& bitmap,
                                    IMAGE_FORMAT format,
                                    int flags /*= 0*/)
{
  /*
  ASSERT(FALSE);
  FIBITMAP* fi = FreeImage_ConvertFromRawBits(const_cast<unsigned char*>(bitmap->Data()), bitmap->Width(), bitmap->Height(), bitmap->Width() * 3, 24, 0,0,0, TRUE);
  int result = FreeImage_Save(MapOwnFormatToFreeImageFormat(format), fi, path.c_str(), JPEG_QUALITYAVERAGE);
  FreeImage_Free(fi);
  return true;*/
  return false;
}

#define FIF2FORMAT(TYPE) case FIF_##TYPE: return FORMAT_##TYPE;
#define FORMAT2FIF(TYPE) case FORMAT_##TYPE: return FIF_##TYPE;

gml::ImageLoader::IMAGE_FORMAT gml::FRImageLoader::MapFreeImageFormatToOwnFormat(FREE_IMAGE_FORMAT f)
{
  switch (f)
  {
      FIF2FORMAT(BMP)
      FIF2FORMAT(ICO)
      FIF2FORMAT(JPEG)
      FIF2FORMAT(PNG)
      FIF2FORMAT(PGM)
      FIF2FORMAT(PGMRAW)
      FIF2FORMAT(PPM)
      FIF2FORMAT(TIFF) 
      FIF2FORMAT(TARGA) 

    default:
      return FORMAT_UNKNOWN;
  }
}

FREE_IMAGE_FORMAT gml::FRImageLoader::MapOwnFormatToFreeImageFormat(gml::ImageLoader::IMAGE_FORMAT f)
{
  switch (f)
  {
      FORMAT2FIF(BMP)
      FORMAT2FIF(ICO)
      FORMAT2FIF(JPEG)
      FORMAT2FIF(PNG)
      FORMAT2FIF(PGM)
      FORMAT2FIF(PGMRAW)
      FORMAT2FIF(PPM)
      FORMAT2FIF(TIFF) 
      FORMAT2FIF(TARGA) 

    default:
      return FIF_UNKNOWN;
  }
}