//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlpathstring.h,v 1.7 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#ifndef GMLPATHSTRING_H_INCLUDED
#define GMLPATHSTRING_H_INCLUDED

/** @file gmlpathstring.h
 *  @brief defines gml::PathString
 */

#include "../base/gmlcommon.h"
#include "../base/gmlstring.h"

#include <list>

namespace gml
{
  using namespace std;

  /** @addtogroup Files
   *  @{
   */
  /// Processing of the path to file
  class PathString : public String
  {
    public:

      static const char PATH_SEPARATOR; //!< char path separator for currents OS
      static const string PATH_SEPARATOR_STR;//!< string path separator for current OS

      /// @name Constructors
      //  @{
      PathString() : String()
      {
      }

      PathString(const string& rhs) : String(rhs)
      {
        /*CorrectSeparators(); */
      }

      PathString(const string& rhs, size_type pos, size_type n) : String(rhs,
                                                                         pos,
                                                                         n)
      {
        /*CorrectSeparators(); */
      }

      PathString(const char* s, size_type n) : String(s, n)
      {
        /*CorrectSeparators(); */
      }

      PathString(const char* s) : String(s)
      {
        /*CorrectSeparators(); */
      }

      PathString(size_type n, char c) : String(n, c)
      {
        /*CorrectSeparators(); */
      }

      PathString(const_iterator first, const_iterator last) : String(first,
                                                                     last)
      {
        /*CorrectSeparators(); */
      }

      /// Constructs an object from path+filename+extension
      /** 
       * The path separator will be added if string path does not end by
       * path separator. The dot will be added between filename and extension
       * if extension does not begin by dot. 
       *
       * @note If parameter extension is absent
       *       then filename contains the full name of the file.
       */
      PathString(const string& path,
                 const string& filename,
                 const string& extension = string());

      //@}

      /*
      PathString& operator=(const char *s)
        {
        String::operator=(s);
        CorrectSeparators(); 
        return *this;
        }
      PathString& operator=(char c)
        {
        string::operator=(c);
        CorrectSeparators(); 
        return *this;
        }
      PathString& operator=(const PathString& rhs)
        {
        String::operator=(rhs);
        CorrectSeparators(); 
        return *this;
        }
        */
      /// @name Path handling
      /// @{

      /// Returns  extracted path from full path to the file.
      /**  @note In case if path cannot be extracted, an empty string
       *         will be returned.
       */
      string Path() const;

      /// Returns full name of the file (with extension). 
      /** @note If full name is not extracted, an empty string will be returned.
       */
      string Filename() const;

      /// Returns base name of the file (without extension)
      /** @note If base name is not extracted, an empty string will be returned.
       */
      string Basename() const;

      /// Returns extension of the file (dot is not included). 
      /** @note If extension is not extracted, an empty string will be returned.
       */
      string Extension() const;

      /// Splits path name into path, base filename and extension.
      /** @note If parameter extension is absent then filename will contain
       *        file name with extension.
       */
      void Split(string* path, string* filename, string* extension = NULL) const;

#if __BORLANDC__ >= 0x550 || !defined(__BORLANDC__)
      /// Expand current path to full path relative to 'relative_to' argument.
      /// @warning relative_to must be a full path or NULL if to work with current directory
      bool MakeFullPath(const PathString& relative_to = PathString());

      /// @}

      /// @name File or directory handling
      /// @{

      /// Check: 'this' string is the name of the real plain file
      bool IsFile() const;


      /// Check: 'this' string is the name of the directory
      bool IsDir() const;

      /// Get current working directory to 'this' pathname
      bool CurDir();
#endif

      /// Change current working directory to 'this' pathname
      bool ChDir() const;

      /// Create directory with name specified by 'this' pathname
      bool MkDir() const;

      /// Remove directory with name specified by 'this' pathname.
      /** @note Directory must be empty
       */
      bool RmDir() const;

      /// Get full path to the current program
      bool ExePath();

      /// Get list of the names of the files and directories which
      /// stored in the directory which defined by the 'this' pathname.
      /** Non-obligatory parameter mask defines mask for the filtration.
       */
      bool ListFileName(list<PathString>& filelist, PathString* mask) const;


      /// @}

    protected:

      bool CheckMask(const PathString& mask) const;
      void CorrectSeparators();
  };

  /** @} */
}
#endif