//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlpathstring.cpp,v 1.9 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#include "../base/gmlcommon.h"
#include "gmlpathstring.h"

#include <vector>
#include <algorithm>

#include <sys/stat.h>
#include <direct.h>

#include <windows.h>

using namespace gml;
using std::vector;

const char gml::PathString::PATH_SEPARATOR = '\\';
const string gml::PathString::PATH_SEPARATOR_STR = string("\\");
const int MAX_PATH_LEN = 500;

PathString::PathString(const string& path,
                       const string& filename,
                       const string& extension)
{
  ASSERT(!filename.empty());

  *this = "";
  *this += path;

  int len = length();
  if (len > 0 && (*this)[len - 1] != PATH_SEPARATOR)
  {
    *this += PATH_SEPARATOR_STR;
  }

  *this += filename;

  if (!extension.empty())
  {
    if (extension[0] != '.')
      *this += ".";
    *this += extension;
  }

  CorrectSeparators();
}


string PathString::Path() const
{
  int pos = find_last_of(PATH_SEPARATOR);
  if (pos == npos)
    return string();

  // Check for logical drive presence
  if (pos > 0 && (*this)[pos - 1] == ':')
    pos++;

  // Check for root path
  if (pos == 0)
    pos++;

  return substr(0, pos);
}

string PathString::Filename() const
{
  int pos = find_last_of(PATH_SEPARATOR);
  if (pos == npos)
    return *this;

  ++pos;
  return substr(pos);
}

string PathString::Basename() const
{
  string fname = Filename();
  int pos = fname.find_last_of('.');
  if (pos == npos)
    return fname;

  return fname.substr(0, pos);
}

string PathString::Extension() const
{
  string fname = Filename();
  int pos = fname.find_last_of('.');
  if (pos == npos)
    return string();

  return fname.substr(pos + 1);
}

void PathString::Split(string* path, string* filename, string* extension) const
{
  if (path != NULL)
    *path = (*this).Path();

  if (filename != NULL)
  {
    if (extension != NULL)
      *filename = (*this).Basename();
    else
      *filename = (*this).Filename();
  }

  if (extension != NULL)
    *extension = (*this).Extension();
}

#if __BORLANDC__ >= 0x550  || !defined(__BORLANDC__)

bool PathString::IsFile() const
{
  struct _stat s;
  int res = _stat(c_str(), &s);
  return res == 0 && (s.st_mode & _S_IFREG);
}

bool PathString::IsDir() const
{
  struct _stat s;
  int res = _stat(c_str(), &s);
  return res == 0 && (s.st_mode & _S_IFDIR);
}


bool PathString::CurDir()
{
  char buffer[255];

  // Receive CWD by getcwd() 
  if (_getcwd(buffer, 255) == NULL)
    return false;

  *this = buffer;
  return true;
}
#endif

bool PathString::ChDir() const
{
  if (empty())
    return false;

  if (chdir(c_str()) != 0)
    return false;

  return true;
}


bool PathString::ExePath()
{
  char buffer[500];
  if (GetModuleFileName(NULL, buffer, 500) == 0)
    return false;
  *this = PathString(buffer);
  return true;
};



bool PathString::ListFileName(list<PathString>& fnames, PathString* mask) const
{
  //ASSERT(isdir());

  fnames.resize(0);
  // ------------------------------------------------
  //             Open catalog
  // ------------------------------------------------

  HANDLE handle = NULL;

  // ------------------------------------------------
  //             Read catalog
  // ------------------------------------------------

  int n_names = 0;

  WIN32_FIND_DATA fdata;
  PathString path(*this, "*.*");  // All items of the catalog
  // for the anologous with UNIX
  // Read first item from catalog list
  handle = FindFirstFile(path.c_str(), &fdata);
  if (handle == INVALID_HANDLE_VALUE)
    return false;

  while (FindNextFile(handle, &fdata))
  {
    // Skip parent and current catalogs
    if ((fdata.cFileName[0] == '.' && fdata.cFileName[1] == '\0') ||
        (fdata.cFileName[0] == '.' &&
         fdata.cFileName[1] == '.' &&
         fdata.cFileName[2] == '\0'))
      continue;

    PathString filename = PathString(fdata.cFileName);

    if (mask != NULL && filename.CheckMask(*mask) != true)
      continue;

    if (n_names == 0)
      (void) fnames.resize(0);

    fnames.push_back(filename);

    n_names++;
  }

  // ------------------------------------------------
  //             Close catalog
  // ------------------------------------------------


  if (!FindClose(handle))
    return false;

  return true;
}


// -------------------------------------------------------------
// NAME         MkDir()
// PURPOSE      Create directory with name  specified by 'this' pathname
// RETURNS      SUCCESS/FAILURE
// NOTE         On base os_mkdir() from IOSL
// -------------------------------------------------------------
bool PathString::MkDir() const
{
  if (this->empty())
    return true;

  if (mkdir(this->c_str()) == 0)
    return false;

  return false;
}


// -------------------------------------------------------------
// NAME         RmDir()
// PURPOSE      Remove directory with name specified by 'this' pathname
// RETURNS      SUCCESS/FAILURE
// NOTE         1. catalog must be empty 
//              2. base is os_rmdir() from IOSL
// -------------------------------------------------------------
bool PathString::RmDir() const
{
  if (this->empty())
    return false;

  if (rmdir(this->c_str()) == 0)
    return true;

  return false;
}
// END OF METHOD PathStr::MkDir()




// -------------------------------------------------------------
//  NAME     RecMatch()
//  PURPOSE  Compare string with mask 
//  RETURNS  1 - string is match with mask
//           0, 2  - strings not match
//  NOTE     codes get from module $(Integra)\tools\zip\util.c
//           funtion recmatch()
// -------------------------------------------------------------
int RecMatch(unsigned char* p, // sh pattern to match 
unsigned char* s)  // string to match it to
{
  unsigned int c;  // pattern char or start of range in [-] loop  

  // Get first character, the pattern for new recmatch calls follows 
  c = *p++;

  // If that was the end of the pattern, match if string empty too 
  if (c == 0)
    return *s == 0;

  // '?' matches any character (but not an empty string) 
  if (c == '?')
    return *s ? RecMatch(p, s + 1) : 0;

  // '*' matches any number of characters, including zero 
  if (c == '*')
  {
    if (*p == 0)
      return 1;
    for (; *s; s++)
      if ((c = RecMatch(p, s)) != 0)
        return c;
    return 2;           // 2 means give up--shmatch will return false 
  }

  // If escape ('\'), just compare next character 
  if (c == '\\')
    if ((c = *p++) == 0)    // if \ at end, then syntax error
      return 0;

  // Just a character--compare it 
#ifdef _WIN32
  if (toupper((int) c) == toupper(*s))
    #else                 // UNIX
    if (c == (*s))
    #endif
      return RecMatch(p, ++s);

  return 0;
}


bool PathString::CheckMask(const PathString& mask) const
{
  const char* p1 = this->c_str();
  const char* p2 = mask.c_str();

  if (RecMatch((unsigned char *) p2, (unsigned char *) p1) != 1)
    return false;

  return true;
}

#if __BORLANDC__ >= 0x550 || !defined(__BORLANDC__)
bool PathString::MakeFullPath(const PathString& relative_to)
{
  gml::PathString cur_path;
  if (!relative_to.empty())
  {
    // save current directory
    cur_path.CurDir();
    // and setup new dir
    bool res = relative_to.ChDir();

    //gml::PathString new_path(relative_to);
    //bool res = new_path.ChDir();

    if (!res)
    {
      cur_path.ChDir(); 
      return false;
    }
  }

  char full_path[MAX_PATH_LEN + 1];
  _fullpath(full_path, this->c_str(), MAX_PATH_LEN);

  *this = full_path;

  return true;
}
#endif

void PathString::CorrectSeparators()
{
  std::replace(begin(), end(), '/', PATH_SEPARATOR);
}