//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlinifile.h,v 1.5 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#ifndef _INIFILE_H_
#define _INIFILE_H_

#include <string>
#include <vector>
#include <istream.h>

#pragma warning (disable : 4786)

namespace gml
{
  /** @addtogroup Files
   *  @{
   */

  class CIniFile
  {
      //all private variables
    private:

      /// Stores pathname of ini file to read/write
      std::string path;

      /// All keys are of this time
      struct key
      {
          /// List of values in key
          std::vector<std::string> values; 

          /// Corresponding list of value names
          std::vector<std::string> names;
      };

      /// List of keys in ini
      std::vector<key> keys; 

      /// Corresponding list of keynames
      std::vector<std::string> names; 


      //all private functions
    private:

      /// Overloaded to take CString
      istream& getline(istream& is, std::string& str);

      /// @return index of specified value, in the specified key, or -1 if not found
      int FindValue(int keynum, const char* valuename);

      /// @return index of specified key, or -1 if not found
      int FindKey(const char* keyname);


      //public variables
    public:

      /// @brief will contain error info if one occurs
      ///        ended up not using much, just in ReadFile and GetValue
      std::string error;

      //public functions
    public:

      inline bool IsSuchKeyInFile(const char* key)
      {
        return (FindKey(key) >= 0);
      };      

      inline bool IsSuchNameInThisKey(const char* key, const char* name)
      {
        return (FindValue(FindKey(key), name) >= 0);
      };

      /// Default constructor
      CIniFile();

      /// Constructor, can specify pathname here instead of using SetPath later
      CIniFile(const char* inipath);

      /// Default destructor
      virtual ~CIniFile();

      /// Sets path of ini file to read and write from
      void SetPath(const char* newpath);

      /// @brief Reads ini file specified using CIniFile::SetPath()
      /// @return true if successful, false otherwise
      bool ReadFile();

      /// Writes data stored in class to ini file
      void WriteFile(); 

      /// Deletes all stored ini data
      void Reset();

      /// @return number of keys currently in the ini
      int GetNumKeys();

      /// @reeturn number of values stored for specified key
      int GetNumValues(const char* keyname);

      /// @brief gets value of [keyname] valuename = 
      /// @return "", or 0 if key/value not found.  
      ///         Sets error member to show problem
      std::string GetValue(const char* keyname, const char* valuename); 

      /// @brief gets value of [keyname] valuename = 
      /// @return "", or 0 if key/value not found.  
      ///         Sets error member to show problem
      void GetValue(const char* keyname,
                    const char* valuename,
                    char* out_pcBuf,
                    int in_iMaxChar);

      /// @brief gets value of [keyname] valuename = 
      /// @return "", or 0 if key/value not found.  
      ///         Sets error member to show problem
      int GetValueI(const char* keyname, const char* valuename); 

      /// @brief gets value of [keyname] valuename = 
      /// @return "", or 0 if key/value not found.  
      ///         Sets error member to show problem
      double GetValueF(const char* keyname, const char* valuename);

      /// @brief sets value of [keyname] valuename =.
      ///
      /// specify the optional paramter as false (0) if you do not want it to create
      /// the key if it doesn't exist. 
      ///
      /// @return  true if data entered, false otherwise
      bool SetValue(const char* key,
                    const char* valuename,
                    const char* value,
                    bool create = 1);
      /// @brief sets value of [keyname] valuename =.
      ///
      /// specify the optional paramter as false (0) if you do not want it to create
      /// the key if it doesn't exist. 
      ///
      /// @return  true if data entered, false otherwise
      bool SetValueI(const char* key,
                     const char* valuename,
                     int value,
                     bool create = 1);
      /// @brief sets value of [keyname] valuename =.
      ///
      /// specify the optional paramter as false (0) if you do not want it to create
      /// the key if it doesn't exist. 
      ///
      /// @return  true if data entered, false otherwise
      bool SetValueF(const char* key,
                     const char* valuename,
                     double value,
                     bool create = 1);

      /// @brief deletes specified value
      /// @return true if value existed and deleted, false otherwise
      bool DeleteValue(const char* keyname, const char* valuename);

      /// @brief deletes specified key and all values contained within
      /// @return returns true if key existed and deleted, false otherwise
      bool DeleteKey(const char* keyname);
  };


  /** @} */
} // end of namespace

#endif
