//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlinifile.cpp,v 1.4 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#pragma warning (disable : 4786)

#include "../base/gmlcommon.h"

#include "gmlinifile.h"
#include "fstream.h"



using namespace gml;

/////////////////////////////////////////////////////////////////////
// Construction/Destruction
/////////////////////////////////////////////////////////////////////

//default constructor
CIniFile::CIniFile()
{
}

//constructor, can specify pathname here instead of using SetPath later
CIniFile::CIniFile(const char* inipath)
{
  path = inipath;
}

//default destructor
CIniFile::~CIniFile()
{
}

/////////////////////////////////////////////////////////////////////
// Public Functions
/////////////////////////////////////////////////////////////////////

//sets path of ini file to read and write from
void CIniFile::SetPath(const char* newpath)
{
  path = newpath;
}

//reads ini file specified using CIniFile::SetPath()
//returns true if successful, false otherwise
bool CIniFile::ReadFile()
{
  ifstream inifile;
  std::string readinfo;
  inifile.open(path.c_str());

  int curkey = -1, curval = -1;
  if (inifile.fail())
  {
    error = "Unable to open ini file.";
    return 0;
  }

  std::string keyname, valuename, value;
  std::string temp;

  while (getline(inifile, readinfo))
  {
    if (readinfo != "")
    {
      if (readinfo[0] == '[' && readinfo[readinfo.length() - 1] == ']') //if a section heading
      {
        keyname = readinfo;
        keyname.erase(0, keyname.find_first_of('[') + 1);
        keyname.erase(keyname.find_last_of(']'), keyname.length());
      }
      else //if a value
      {
        valuename = readinfo.substr(0, readinfo.find_first_of('='));
        value = readinfo.substr(readinfo.find_first_of('=') + 1,
                                readinfo.length());
        SetValue(keyname.c_str(), valuename.c_str(), value.c_str());
      }
    }
  }
  inifile.close();
  return 1;
}

//writes data stored in class to ini file
void CIniFile::WriteFile()
{
  ofstream inifile;
  inifile.open(path.c_str());
  for (int keynum = 0; keynum <= names.size() - 1; keynum++)
  {
    if (keys[keynum].names.size() != 0)
    {
      inifile << '[' << names[keynum].c_str() << ']' << endl;
      for (int valuenum = 0;
           valuenum <= keys[keynum].names.size() - 1;
           valuenum++)
      {
        inifile << keys[keynum].names[valuenum].c_str() << "="
                << keys[keynum].values[valuenum].c_str();
        if (valuenum != keys[keynum].names.size() - 1)
          inifile << endl;
        else if (keynum < names.size())
          inifile << endl;
      }
      if (keynum < names.size())
        inifile << endl;
    }
  }
  inifile.close();
}

//deletes all stored ini data
void CIniFile::Reset()
{
  keys.clear();
  names.clear();
}

//returns number of keys currently in the ini
int CIniFile::GetNumKeys()
{
  return keys.size();
}

//returns number of values stored for specified key, or -1 if key found
int CIniFile::GetNumValues(const char* keyname)
{
  int keynum = FindKey(keyname);
  if (keynum == -1)
    return -1;
  else
    return keys[keynum].names.size();
}

//gets value of [keyname] valuename = 
//overloaded to return std::string, int, and double
std::string CIniFile::GetValue(const char* keyname, const char* valuename)
{
  int keynum = FindKey( keyname), valuenum = FindValue(keynum, valuename);

  if (keynum == -1)
  {
    error = "Unable to locate specified key.";
    return "";
  }

  if (valuenum == -1)
  {
    error = "Unable to locate specified value.";
    return "";
  }
  return keys[keynum].values[valuenum];
}


//===================================================================
//= Function name : CIniFile::GetValue
//= Description   : 
//= Return type   : void 
//===================================================================

void CIniFile::GetValue(const char* keyname,
                        const char* valuename,
                        char* out_pcBuf,
                        int in_iMaxChar)
{
  std::string TmpStr = GetValue(keyname, valuename);
  strncpy(out_pcBuf, TmpStr.c_str(), in_iMaxChar);
}

//gets value of [keyname] valuename = 
//overloaded to return std::string, int, and double
int CIniFile::GetValueI(const char* keyname, const char* valuename)
{
  return atoi(GetValue(keyname, valuename).c_str());
}

//gets value of [keyname] valuename = 
//overloaded to return std::string, int, and double
double CIniFile::GetValueF(const char* keyname, const char* valuename)
{
  return atof(GetValue(keyname, valuename).c_str());
}

//sets value of [keyname] valuename =.
//specify the optional paramter as false (0) if you do not want it to create
//the key if it doesn't exist. Returns true if data entered, false otherwise
//overloaded to accept std::string, int, and double
bool CIniFile::SetValue(const char* keyname,
                        const char* valuename,
                        const char* value,
                        bool create)
{
  int keynum = FindKey( keyname), valuenum = 0;
  //find key
  if (keynum == -1) //if key doesn't exist
  {
    if (!create) //and user does not want to create it,
      return 0; //stop entering this key    

    keys.resize(keys.size() + 1);
    keynum = names.size();
    names.push_back(keyname);
  }

  //find value
  valuenum = FindValue(keynum, valuename);
  if (valuenum == -1)
  {
    if (!create)
      return 0;
    keys[keynum].names.resize(keys[keynum].names.size() + 1);
    keys[keynum].values.resize(keys[keynum].names.size() + 1);
    valuenum = keys[keynum].names.size() - 1;
    keys[keynum].names[valuenum] = valuename;
  }
  keys[keynum].values[valuenum] = value;
  return 1;
}

//sets value of [keyname] valuename =.
//specify the optional paramter as false (0) if you do not want it to create
//the key if it doesn't exist. Returns true if data entered, false otherwise
//overloaded to accept std::string, int, and double
bool CIniFile::SetValueI(const char* keyname,
                         const char* valuename,
                         int value,
                         bool create)
{
  char cBuf[100];
  sprintf(cBuf, "%d", value);
  return SetValue(keyname, valuename, cBuf, create);
}

//sets value of [keyname] valuename =.
//specify the optional paramter as false (0) if you do not want it to create
//the key if it doesn't exist. Returns true if data entered, false otherwise
//overloaded to accept std::string, int, and double
bool CIniFile::SetValueF(const char* keyname,
                         const char* valuename,
                         double value,
                         bool create)
{
  char cBuf[100];
  sprintf(cBuf, "%e", value);
  return SetValue(keyname, valuename, cBuf, create);
}

//deletes specified value
//returns true if value existed and deleted, false otherwise
bool CIniFile::DeleteValue(const char* keyname, const char* valuename)
{
  int keynum = FindKey( keyname), valuenum = FindValue(keynum, valuename);
  if (keynum == -1 || valuenum == -1)
    return 0;

  keys[keynum].names.erase(keys[keynum].names.begin() + valuenum);
  keys[keynum].values.erase(keys[keynum].values.begin() + valuenum);
  return 1;
}

//deletes specified key and all values contained within
//returns true if key existed and deleted, false otherwise
bool CIniFile::DeleteKey(const char* keyname)
{
  int keynum = FindKey(keyname);
  if (keynum == -1)
    return 0;

  keys.erase(keys.begin() + keynum);
  names.erase(names.begin() + keynum);

  return 1;
}

/////////////////////////////////////////////////////////////////////
// Private Functions
/////////////////////////////////////////////////////////////////////

//returns index of specified key, or -1 if not found
int CIniFile::FindKey(const char* keyname)
{
  int keynum = 0;
  while (keynum < keys.size() && names[keynum] != keyname)
  {
    keynum++;
  }
  if (keynum == keys.size())
    return -1;
  return keynum;
}

//returns index of specified value, in the specified key, or -1 if not found
int CIniFile::FindValue(int keynum, const char* valuename)
{
  if (keynum == -1)
    return -1;
  int valuenum = 0;
  while (valuenum < keys[keynum].names.size() &&
         keys[keynum].names[valuenum] != valuename)
  {
    valuenum++;
  }
  if (valuenum == keys[keynum].names.size())
    return -1;
  return valuenum;
}

//overloaded from original getline to take std::string
istream& CIniFile:: getline(istream& is, std::string& str)
{
  char buf[2048];
  is.getline(buf, 2048);
  str = buf;
  return is;
}
