//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlfile.h,v 1.5 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#ifndef _GMLFILE_H_INCLUDED
#define _GMLFILE_H_INCLUDED

#include "gmlpathstring.h"

namespace gml
{
  /** @addtogroup Files
   *  @{
   */

  /// A class for working with files (a wrapped to FILE *)
  class File
  {
    public:

      /// @name Constructors
      /// @{

      /// Sets a full path to the file only.
      File(const PathString& full_pathname);

      /// Copy constructor. 
      /** %File may not be opened.
        * Debug version asserts if this condition is violated.
        */
      File(const File& file);

      /// Destructor. File will be closed if it was opened.
      ~File();

      /// @}

      /// @name Methods
      /// @{

      /// Opening of file according to mode. The list of allowed modes:
      /** @param mode - i/o mode of the file to open
        *    - "r"  - open for reading only; the file must exist.
        *    - "w"  - create a new file or open an existing file for writing.
        *    - "a"  - open for writing at the end-of-file or create for writing
        *            if the file does not exist.
        *    - "r+" - open an existing file for reading and writing.
        *    - "w+" - open an empty file for reading and writing.
        *    - "a+" - open for reading and appending; create a file if it does
        *       not exist.
        * @note
        * Files are open in text mode as a default. Character 'b' may be
        * appended to the above modes to open file in binary mode ('b' may
        * also precede '+' sign).
        * Path to file must be specified. Debug version asserts if this
        * condition is violated.
        */
      bool Open(const char* mode);

      /// True if file was previously opened
      bool Opened() const
      {
        return (fd != NULL);
      }

      /// Close the file.
      /**
        * @return 
        *     - true (if there were no problems when working with the
        *       file and the file was closed successfully). 
        *     - false (either I/O error while working with file or closing error).
        * @note
        *  %File must be opened. Debug version asserts if this condition is
        *  violated
        */
      bool Close();


      // Reads string from the file (opened as text) to buffer out_buff with
      // specified size len_buff. The new line character will NOT be put into
      // out_buff. If the buffer is not big enough to place the next string
      // then the next len_buff - 1 characters are read into the buffer
      // followed by '\0'.
      // Optional out_nl flag indicates whether newline character was
      // actually read or not.
      // Returns: SUCCESS or FAILURE (reading error).
      // File must be opened on reading in text mode. Debug version asserts it.
      // out_buff may not be NULL and len_buff must be  > 0.
      // Debug version asserts it.
      bool ReadStr(String& out_buff, int len_buff, bool* out_nl = NULL);
      bool ReadStr(char* out_buff, int len_buff, bool* out_nl = NULL);
      bool ReadStr(String& out_buff);
      // Reads len_buff bytes from the file (opened as binary) to duffer
      // out_buff. out_size will be set by  number of read characters.
      // Returns: SUCCESS or FAILURE (reading error).
      // File must be opened on reading in binary mode.
      // Debug version asserts it.
      // out_buff may not be NULL and len_buff must be  > 0.
      // Debug version asserts it.
      void Read(BYTE* out_buff, int len_buff, int& out_size);

      // Write string str followed by new line character to the file.
      // File must be opened on writing in the text mode.
      // Debug version asserts it.
      void WriteStr(const String& str);

      // Writes string to the file (opened as text) from out_buff
      // The new line character will be written to the file.
      // File must be opened on writing in the text mode.
      // Debug version asserts it.
      void WriteStr(const char* out_buff, ...);

      // Writes size bytes to the file (opened as binary) from out_buff.
      // File must be opened on writing in the binary mode.
      // Debug version asserts it.
      // out_buff may not be NULL and size must be >= 0.
      // Debug version asserts it.
      void Write(const BYTE* buff, int size);

      // Flushes a stream of the file.
      // File must be opened. Debug version asserts it.
      void Flush();

      // Deletes file, previously, file will be closed if it was opened.
      // File may not be opened. Debug version asserts it.
      bool Remove();

      // Renames a file to new_file_name, previously, file will be closed
      // if it was opened..
      // File may not be opened. Debug version asserts it.
      bool Rename(const PathString& new_file_name);

      // Detects the error of file operation.
      // Returns: Result of detection.
      bool IsError();

      // Returns full pathname to the file.
      PathString PathName() const
      {
        return name;
      }

      // Formatted output to this file (analog of fprintf function),
      // about allowed formats please see Portability Principles).
      int Printf(const char* format, ...);

      // Copy file with name 'name_from' to the file with name 'name_to'
      static bool Copy(const PathString& name_from, const PathString& name_to);

      // Move the file pointer to a new location that is
      // 'offset' bytes from the beginning of file
      bool Seek(long offset);

      // Move the file pointer to a new location that is
      // 'offset' bytes from the current position of file pointer
      bool SeekCur(long offset);

      // Move the file pointer to a new location that is
      // 'offset' bytes from the end of file
      bool SeekEnd(long offset);

      /// Gets current file offset
      long FilePos() const;

      /// Returns the size of the file in bytes
      long FileSize() const;
      // Class members

    private:

      PathString name;  // full pathname/filename.extension

    protected:

      FILE* fd;      // file descriptor

      enum { MODE_TEXT, MODE_BINARY };
      int mode_tb;

      enum { MODE_READ, MODE_WRITE, MODE_READ_WRITE };
      int mode_rw;
  };
  /** @} */
}

#endif
