//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlfile.cpp,v 1.7 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

// WIN32
#ifdef _WIN32
#include <windows.h>
#include <io.h>
#include <cstdio>
#endif

// SYSTEM INCLUDES
#include <stdarg.h>
#include <locale.h>

#include "../base/gmlcommon.h"
#include "gmlfile.h"



using namespace gml;

File::File(const PathString& full_pathname) : name(full_pathname)
{
  fd = NULL;
}

File::File(const File& sour) : name(sour.name)
{
  ASSERT(sour.fd == NULL);

  fd = NULL;
}
// END OF FUNCTION File::File()

// ----------------------------------------------------
// NAME        ~File
// PURPOSE     Destructor
// PARAMETERS  none.
// RETURN      none.
// NOTES       File will be closed if it was be opened.
// ----------------------------------------------------
File::~File()
{
  if (fd != NULL)
    Close();
}
// END OF FUNCTION File::~File()

// -------------------------------------------------------------------------
// NAME        Open()
// PURPOSE     Opening of file according to mode.
// PARAMETERS  const char *mode - mode of
//             the opening file
// RETURN      true or false (opening error).
// EXTERNS     none.
// NOTES       The list of allowed modes:
//             "r"  - open for reading only; the file must exist.
//             "w" - create a new file or open an existing file for writing.
//             "a" - open for writing at the end-of-file or create for
//                    writing if the file does not exist.
//             "r+" - open an existing file for reading and writing.
//             "w+" - open an empty file for reading and writing.
//             "a+" - open for reading and appending; create a file if it
//                    does not exist.
//             Files are open in text mode as a default. Character 'b' may be
//             appended to the above modes to open file in binary mode ('b'
//             may also precede '+' sign).
//             Path to file must be specified. Debug version asserts if this
//             condition is violated.
// -------------------------------------------------------------------------
bool File::Open(const char* mode)
{
  char open_mode[4];
  int i = 0;

  // File must be closed
  ASSERT(fd == NULL);

  // Name of the file must be defined
  ASSERT(!name.empty());

  // First symbol from fixed range
  ASSERT(mode[0] == 'r' || mode[0] != 'w' || mode[0] != 'a');

#ifdef _DEBUG
  if (mode[0] == 'r')
    mode_rw = MODE_READ;
  else // mode[0] == 'w' or mode[0] == 'a'
    mode_rw = MODE_WRITE;
#endif

  i = 1;
  if (mode[1] == '+')
  {
#ifdef _DEBUG
    mode_rw = MODE_READ_WRITE;
#endif
    i = 2;
  }

  if (mode[i] == '\0')
  {
#ifdef _DEBUG
    mode_tb = MODE_TEXT;
#endif

    strcpy(open_mode, mode); // "r", "w", "a", "r+", "w+", "a+"

#ifdef _WIN32
    open_mode[i++] = 't';
    open_mode[i] = '\0';
#endif
  }  // if (mode[i] == '\0')
  else
  {
#ifdef _DEBUG
    mode_tb = MODE_BINARY;
#endif

    ASSERT(mode[i] == 'b');

#ifdef _WIN32
    ASSERT(mode[i + 1] == '\0' || (mode[i + 1] == '+' && mode[i + 2] == '\0'));
#else
    ASSERT(mode[i + 1] != '\0');
#endif

    strcpy(open_mode, mode);
  }  // if (mode[i] != '\0')

  fd = fopen((char *) name.data(), open_mode);
  if (fd == NULL)
    return false;

  return true;
}
// END OF FUNCTION File::Open()

// ------------------------------------------------------------------------
// NAME         Close()
// PURPOSE      Closes the file.
// ARGUMENTS    none.
// RETURNS      true (if there were no problems when working
//                       with the file and the file was closed truefully)
//              false (either I/O error while working with file or closing
//                       error).
// EXTERNS      none.
// NOTES        ANSI fclose() compatible
//              File must be opened. Debug version asserts if this condition
//              is violated.
// -------------------------------------------------------------------------
bool File::Close()
{
  ASSERT(fd != NULL);      // file must be opened

  int err1 = ferror(fd);  // remember if errors occurred
  int err2 = fclose(fd);  // remember close errors
  fd = NULL;

  return ((err1 || err2) ? false : true);
}
// END OF FUNCTION File::Close()

// ------------------------------------------------------
//                   Read from file
// ------------------------------------------------------

bool File::ReadStr(char* out_buff, int len_buff, bool* out_nl)
{
  ASSERT(fd != NULL);
  ASSERT(mode_tb == MODE_TEXT && mode_rw != MODE_WRITE);
  ASSERT(out_buff != NULL && len_buff > 0);

  if (fgets(out_buff, len_buff, fd) == NULL)
    return false;

  // Remove NL character if any
  if (out_nl != NULL)
    *out_nl = FALSE;
  for (char*pc = out_buff; *pc != '\0'; pc++)
  {
    if (*pc == '\n' || *pc == '\r')
    {
      *pc = '\0';
      if (out_nl != NULL)
        *out_nl = TRUE;
      break;
    }
  }

  return true;
}

bool File::ReadStr(String& out_buff, int len_buff, bool* out_nl)
{
  ASSERT(fd != NULL);
  ASSERT(mode_tb == MODE_TEXT && mode_rw != MODE_WRITE);
  ASSERT(len_buff > 0);

  char* new_str = new char[len_buff + 1];

  bool res = ReadStr(new_str, len_buff, out_nl);

  if (res == true)
    out_buff = new_str;

  delete[] new_str;

  return res;
}

bool File::ReadStr(String& out_buff)
{
  const int buf_size = 128;
  char* buffer = new char[buf_size + 1];

  out_buff.erase();
  for (; ;)
  {
    if (fgets(buffer, buf_size, fd) == NULL)
    {
      delete[] buffer;
      return feof(fd) ? false : true;
    }
    out_buff += buffer;
    if (out_buff[out_buff.size() - 1] == '\n') // we've read line till the end
    {
      out_buff.erase(out_buff.size() - 1); // delete \n
      delete[] buffer;
      return true;
    }
  }
  return true;
}
// -------------------------------------------------------------------
// NAME         Read
// PURPOSE      Reads len_buff bytes from the file (opened as binary)
//              to duffer out_buff. out_size will be set by number of
//              read characters.
// ARGUMENTS    BYTE *out_buf - buffer for the data
//              int len_buff  - length of the buffer
//              int &out_size - length of the read data
// RETURNS      none.
// EXTERNS      none.
// NOTES        1. File must beopened on reading in binary mode.
//                 Debug version asserts it.
//              2. out_buff may not be NULL and len_buff must be > 0.
//                 Debug version asserts it.
// --------------------------------------------------------------------
void File::Read(BYTE* out_buff, int len_buff, int& out_size)
{
  ASSERT(fd != NULL);
  ASSERT(mode_tb == MODE_BINARY && mode_rw != MODE_WRITE);
  ASSERT(out_buff != NULL && len_buff >= 0);

  out_size = fread(out_buff, sizeof(BYTE), len_buff, fd);

  return;
}
// END OF FUNCTION Read()

// -----------------------------------------------------------------
//                  Write to the file
// -----------------------------------------------------------------

// -------------------------------------------------------------------------
// NAME         WriteStr
// PURPOSE      Write string str followed by new line character to the file.
// ARGUMENTS    const Str &str - writing string
// RETURNS      true/false (writing error).
// EXTERNS      none.
// NOTES        File must be opened on writing in the text mode.
//              Debug version asserts it.
// ------------------------------------------------------------------------
void File::WriteStr(const String& str)
{
  ASSERT(fd != NULL);
  ASSERT(!str.empty());
  ASSERT(mode_tb == MODE_TEXT && mode_rw != MODE_READ);

  WriteStr(str.c_str());
  return;
}
// END OF FUNCTION WriteStr(const Str &str)

// ----------------------------------------------------------------------
// NAME         WriteStr
// PURPOSE      Writes string to the file (opened as text) from out_buff
//              with specified length len_buff. The new line character will
//              be written to the file.
// ARGUMENTS    const char *out_buff - buffer
// RETURNS      none.
// EXTERNS      none.
// NOTES        1. File must be opened on writing in the text mode.
//                 Debug version asserts it.
//              2. out_buff may not be NULL and len_buff must be > 0.
//                 Debug version asserts it.
// -----------------------------------------------------------------------
void File::WriteStr(const char* out_buff, ...)
{
  va_list arg_list;
  va_start(arg_list, out_buff);

  ASSERT(fd != NULL);
  ASSERT(mode_tb == MODE_TEXT && mode_rw != MODE_READ);
  ASSERT(out_buff != NULL);

  vfprintf(fd, out_buff, arg_list);
  fputs("\n", fd);

  va_end(arg_list);
}
// END OF FUNCTION WriteStr(const char *out_buff)

// -----------------------------------------------------------------
// NAME         Write
// PURPOSE      Writes size bytes to the file (opened as binary)
//              from out_buff.
// ARGUMENTS    const Str &str - writing string
// RETURNS      none.
// EXTERNS      none.
// NOTES        1. File must beopened on writing in the binary mode.
//                 Debug version asserts it.
//              2. out_buff may not be NULL and size must be >= 0.
//                 Debug version asserts it.
// ------------------------------------------------------------------
void File::Write(const BYTE* out_buff, int size)
{
  ASSERT(fd != NULL);
  ASSERT(mode_tb == MODE_BINARY && mode_rw != MODE_READ);
  ASSERT(out_buff != NULL && size >= 0);

  fwrite(out_buff, sizeof(BYTE), size, fd);
  return;
}
// END OF FUNCTION File::Write(const BYTE *out_buff, int size)

// -----------------------------------------------------------
// NAME         Flush()
// PURPOSE      Flushes a stream of the file.
// ARGUMENTS    none.
// RETURNS      none.
// EXTERNS      none.
// NOTES        File must beopened. Debug version asserts it.
// -----------------------------------------------------------
void File::Flush()
{
  ASSERT(fd != NULL);

  fflush(fd);
}
// END OF FUNCTION File::Flush()

// --------------------------------------------------------------
// NAME         Remove()
// PURPOSE      Deletes file, previously, file will be closed
//              if it was opened.
// ARGUMENTS    none.
// RETURNS      true / false
// EXTERNS      none.
// NOTES        ANSI remove() compatible
//              File may not be opened. Debug version asserts it.
// --------------------------------------------------------------
bool File::Remove()
{
  ASSERT(fd == NULL);
  ASSERT(!name.empty());

  if (remove(name.data()) != 0)
    return false;

  return true;
}
// END OF FUNCTION Remove()

// -------------------------------------------------------------
// NAME         Rename()
// PURPOSE      Renames a file to new_file_name, previously, file
//              will be closed if it was opened..
// ARGUMENTS    PathString &new_file_name - new name of the file
// RETURNS      true / false
// EXTERNS      none.
// NOTES        This function can be used to move files between
//              directories (but not between drives)
//              Directories cannot be moved.
//              ANSI rename() compatible
//              File may not be opened. Debug version asserts it.
// -------------------------------------------------------------
bool File::Rename(const PathString& new_file_name)
{
  ASSERT(fd == NULL);
  ASSERT(!name.empty() && !new_file_name.empty());

  if (rename(name.data(), new_file_name.data()) != 0)
    return false;

  name = new_file_name;
  return true;
}
// END OF FUNCTION Rename()

// --------------------------------------------------
// NAME         IsError()
// PURPOSE      Detects the error of file operation.
// ARGUMENTS    none.
// RETURNS      Result of detection (TRUE if error).
// EXTERNS      none.
// NOTES        none.
// --------------------------------------------------
bool File::IsError()
{
  ASSERT(fd != NULL);
  return ferror(fd) != 0;
}
// END OF FUNCTION File::IsError()


// -------------------------------------------------------------------
//  NAME         Printf()
//  PURPOSE      Formatted output to this file (analog of fprinf
//               function)
//  ARGUMENTS    const CHAR *str_format, ... format string
//  RETURNS      none.
//  EXTERNS      none.
//  NOTES        fd and str_format must be not NULL. Debug version
//               assert these.
// -------------------------------------------------------------------
int File::Printf(const CHAR* str_format, ...)
{
  ASSERT(fd != NULL);
  ASSERT(str_format != NULL);

  va_list arg_list;
  va_start(arg_list, str_format);

  int res = vfprintf(fd, (const char*) str_format, arg_list);

  va_end(arg_list);

  return res;
}
// END OF FUNCTION File::Printf()

// -------------------------------------------------------------------
//  NAME         Copy()
//  PURPOSE      Copy files
//  ARGUMENTS    PathString &name_from - name of the source file
//               PathString &name_to   - name of the target file
//  RETURNS      true/false
//  EXTERNS      none.
//  NOTES        Corrected by Pnd '00.02.16
// -------------------------------------------------------------------
bool File::Copy(const PathString& name_from, const PathString& name_to)
{
#ifdef _WIN32

  if (!CopyFile(name_from.c_str(), name_to.c_str(), FALSE))
    return false;

#else                  // UNIX

  File fl_from(name_from);
  File fl_to(name_to);

  if (fl_from.Open("rb") != true)
    return false;

  if (fl_to.Open("wb") != true)
    return false;

  const int BUF_LEN = 1000;
  BYTE buff[BUF_LEN];  // currently at stack
  for (; ;)
  {
    int len = 0;
    fl_from.Read(buff, BUF_LEN, len);
    if (len <= 0)
      break;
    fl_to.Write(buff, len);
  }

  if ((fl_from.Close() != true) | (fl_to.Close() != true))
  {
    fl_to.Remove();
    return false;
  }

#endif

  return true;
}

// -------------------------------------------------------------------
//  NAME         Seek(), SeekCur(), SeekEnd()
//  PURPOSE      Move the file pointer to a new location that is
//               'offset' bytes from the beginning of file (Seek) or
//               from the current position of file pointer (SeekCur) or
//               from the end of file (SeekEnd)
//  ARGUMENTS    long offset - specification of the new position of the
//                             file pointer
//  RETURNS      true/false
//  NOTES        None
// -------------------------------------------------------------------
bool File::Seek(long offset)
{
  ASSERT(fd != NULL);
  ASSERT(mode_tb == MODE_BINARY);
  if (fseek(fd, offset, SEEK_SET) == 0)
    return true;
  return false;
}

bool File::SeekCur(long offset)
{
  ASSERT(fd != NULL);
  ASSERT(mode_tb == MODE_BINARY);
  if (fseek(fd, offset, SEEK_CUR) == 0)
    return true;
  return false;
}

bool File::SeekEnd(long offset)
{
  ASSERT(fd != NULL);
  ASSERT(mode_tb == MODE_BINARY);
  if (fseek(fd, offset, SEEK_END) == 0)
    return true;
  return false;
}

long File::FilePos() const
{
  ASSERT(fd != NULL);
  ASSERT(mode_tb == MODE_BINARY);
  return ftell(fd);
}

long File::FileSize() const
{
  //FIXME: slow approach, use file info functions
  long cur_pos = FilePos();
  fseek(fd, 0, SEEK_END);
  int size = FilePos();
  fseek(fd, cur_pos, SEEK_SET);
  return size;
}





