//
//  GMlib -- Graphics & Media Lab Common Source Library
//
//  $Id: gmlcolor.h,v 1.12 2004/01/13 17:38:42 04a_deg Exp $
//
//  Copyright (C) 2004, Moscow State University Graphics & Media Lab
//  gmlsupport@graphics.cs.msu.su
//  
//  This file is part of GMlib software.
//  For conditions of distribution and use, see the accompanying README file.

#ifndef _GMLCOLOR_H_
#define _GMLCOLOR_H_

#include "../base/gmlcommon.h"

namespace gml
{
  /** @addtogroup Color
   *  @{
   */

  /// 3 Color repsentation
  template <class T>
  class TColor3
  {
    public:
      /// no initialization
      TColor3()
      {
      }
      TColor3(T r0, T g0, T b0) : r(r0), g(g0), b(b0)
      {
      }

      inline TColor3<T>& operator *=(const double d)
      {
        x = (T) (x * d); y = (T) (y * d); z = (T) (z * d);return *this;
      }

      inline TColor3<T> operator *(const double d) const
      {
        return TColor3<T>(x * d, y * d, z * d);
      }

      inline TColor3<T>& operator +=(const TColor3<T>& u)
      {
        x += u.x; y += u.y; z += u.z; return *this;
      }

      inline TColor3<T>& operator -=(const TColor3<T>& u)
      {
        x -= u.x; y -= u.y; z -= u.z; return *this;
      }

      inline TColor3<T> operator +(const TColor3<T>& u) const
      {
        return TColor3<T>(x + u.x, y + u.y, z + u.z);
      }

      inline TColor3<T> operator -(const TColor3<T>& u) const
      {
        return TColor3<T>(x - u.x, y - u.y, z - u.z);
      }

      inline TColor3<T>& operator /=(const double d)
      {
        ASSERT(d != (T) 0.0);
        x = (T) (x / d);
        y = (T) (y / d);
        z = (T) (z / d);
        return *this;
      }

      inline TColor3<T> operator /(const double d) const
      {
        ASSERT(d != (T) 0); return TColor3<T>(x / d, y / d, z / d);
      }
      /// Convert a TColor3 to array of (3) elements
      inline operator T *()
      {
        return c;
      }

#ifndef __BORLANDC__
      /// Convert a TColor3 to array of (3) elements
      inline operator const T *() const
      {
        return c;
      }
#endif

      /// Treat array of (3) elements as a TColor3
      inline static const TColor3<T>& Cast(const T* u)
      {
        return *(const TColor3<T>*) u;
      }
      /// Treat array of (3) elements as a TColor3
      inline static TColor3<T>& Cast(T* u)
      {
        return *(TColor3<T>*) u;
      }

    public:
      union
      {
          struct
          {
              T c[3];
          };
          struct
          {
              T r, g, b;
          };
          struct
          {
              T x, y, z;
          };
      };
  };

  /// Multiplication of scalar d by TColor3 u
  /** @relates TColor3
  */
  template <class T>
  inline TColor3<T> operator *(const double d, const TColor3<T>& u)
  {
    return u * d;
  }

  /// 4 Color repsentation
  template <class T>
  class TColor4
  {
    public:
      /// no initialization
      TColor4()
      {
      }
      TColor4(T r0, T g0, T b0, T w0 = 1) : r(r0), g(g0), b(b0), w(w0)
      {
      }

      inline TColor4<T>& operator *=(const double d)
      {
        x = (T) (x * d); y = (T) (y * d); z = (T) (z * d);return *this;
      }

      inline TColor4<T> operator *(const double d) const
      {
        return TColor4<T>(x * d, y * d, z * d);
      }

      inline TColor4<T>& operator +=(const TColor4<T>& u)
      {
        x += u.x; y += u.y; z += u.z; w += u.w; return *this;
      }

      inline TColor4<T>& operator -=(const TColor4<T>& u)
      {
        x -= u.x; y -= u.y; z -= u.z; w -= u.w; return *this;
      }

      inline TColor4<T> operator +(const TColor4<T>& u) const
      {
        return TColor4<T>(x + u.x, y + u.y, z + u.z, w + u.w);
      }

      inline TColor4<T> operator -(const TColor4<T>& u) const
      {
        return TColor4<T>(x - u.x, y - u.y, z - u.z, w - u.w);
      }

      inline TColor4<T>& operator /=(const double d)
      {
        ASSERT(d != (T) 0.0);
        x = (T) (x / d);
        y = (T) (y / d);
        z = (T) (z / d);
        w = (T) (w / d);
        return *this;
      }



      inline TColor4<T> operator /(const double d) const
      {
        ASSERT(d != (T) 0); return TColor4<T>(x / d, y / d, z / d, w / d);
      }

      /// Convert a TColor4 to array of (4) elements
      inline operator T *()
      {
        return c;
      }
      /// Convert a TColor4 to array of (4) elements
      inline operator const T *() const
      {
        return c;
      }

      /// Treat array of (4) elements as a TColor4
      inline static const TColor4<T>& Cast(const T* u)
      {
        return *(const TColor4<T>*) u;
      }
      /// Treat array of (4) elements as a TColor4
      inline static TColor4<T>& Cast(T* u)
      {
        return *(TColor4<T>*) u;
      }

    public:
      union
      {
          struct
          {
              T c[4];
          };
          struct
          {
              T r, g, b, a;
          };
          struct
          {
              T x, y, z, w;
          };
      };
  };

  /// Multiplication of scalar d by TColor4 u
  /** @relates TColor4
  */
  template <class T>
  inline TColor4<T> operator *(const double d, const TColor4<T>& u)
  {
    return u * d;
  }

  typedef TColor3<BYTE> Color3ub;
  typedef TColor3<WORD> Color3w;
  typedef TColor3<short> Color3s;
  typedef TColor3<float> Color3f;
  typedef TColor4<BYTE> Color4ub;
  typedef TColor4<float> Color4f;

  // definitions for old TColorRGB<> & TColorRGBA<>  

  /// @deprecated use Color3 instead
  typedef TColor3<BYTE> ColorRGBub;
  /// @deprecated use Color3 instead
  typedef TColor3<WORD> ColorRGBw;
  /// @deprecated use Color3 instead
  typedef TColor3<float> ColorRGBf;
  /// @deprecated use Color3 instead
  typedef TColor4<BYTE> ColorRGBAub;
  /// @deprecated use Color3 instead
  typedef TColor4<float> ColorRGBAf;


  /** @} */
} // namespace gml

#endif
